# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Dict, List
import logging

from travel.hotels.content_manager.data_model.storage import StorageHotelWL, StoragePermalinkWL
from travel.hotels.content_manager.data_model.stage import WLNewHotelsHotelData, WLNewHotelsPermalinkData
from travel.hotels.content_manager.data_model.types import ClusterizationStartReason, WLResult
from travel.hotels.content_manager.lib.common import dc_from_dict
from travel.hotels.content_manager.lib.dispatcher import Dispatcher
from travel.hotels.content_manager.lib.persistence_manager import Condition
from travel.hotels.content_manager.lib.storage import HotelWLKey, Storage
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)


class UpdaterWLClusterizedHotels(Updater):

    def get_hotels(self, output_path: str) -> Dict[int, List[StorageHotelWL]]:
        hotels = dict()

        hotels_table = self.persistence_manager.join(output_path, 'hotels')
        LOG.info(f'Reading hotels from {hotels_table}')
        raw_hotels = self.persistence_manager.read(hotels_table)

        for raw_hotel in raw_hotels:
            data: WLNewHotelsHotelData = dc_from_dict(WLNewHotelsHotelData, raw_hotel)
            LOG.info(data)
            hotel = StorageHotelWL(
                permalink=data.permalink,
                partner_id=data.partner_id,
                original_id=data.original_id,
                grouping_key=data.grouping_key,
            )
            permalink_hotels = hotels.setdefault(hotel.permalink, list())
            permalink_hotels.append(hotel)

        return hotels

    def get_permalinks(self, permalinks_table: str) -> List[StoragePermalinkWL]:
        permalinks = list()

        LOG.info(f'Reading permalinks from {permalinks_table}')
        raw_permalinks = self.persistence_manager.read(permalinks_table)

        for raw_permalink in raw_permalinks:
            data: WLNewHotelsPermalinkData = dc_from_dict(WLNewHotelsPermalinkData, raw_permalink)
            LOG.info(data)
            permalink = StoragePermalinkWL(
                permalink=data.permalink,
                grouping_key=data.grouping_key,
            )
            permalinks.append(permalink)

        return permalinks

    def run(self, output_path: str, temp_dir: str) -> None:
        local_storage = Storage()

        hotels = self.get_hotels(output_path)

        success_permalinks_table = self.persistence_manager.join(output_path, 'permalinks')
        success_permalinks = self.get_permalinks(success_permalinks_table)

        failed_permalinks_table = self.persistence_manager.join(output_path, 'failed_permalinks')
        failed_permalinks = self.get_permalinks(failed_permalinks_table)

        permalink_ids = set()
        for permalink in success_permalinks + failed_permalinks:
            permalink_ids.add(permalink.permalink)

        LOG.info(f'Getting permalinks from {self.path_info.storage_permalinks_wl_table}')
        raw_permalinks = self.persistence_manager.request_select(
            src_path=self.path_info.storage_permalinks_wl_table,
            dc=StoragePermalinkWL,
            match_conditions=[Condition('permalink', 'in', permalink_ids)],
        )
        LOG.info(f'Getting hotels from {self.path_info.storage_hotels_wl_table}')
        raw_hotels = self.persistence_manager.request_select(
            src_path=self.path_info.storage_hotels_wl_table,
            dc=StorageHotelWL,
            match_conditions=[Condition('permalink', 'in', permalink_ids)],
        )

        # workaround for old ClusterizationStartReason values
        allowed_start_reasons = [r.value for r in ClusterizationStartReason]
        for raw_permalink in raw_permalinks:
            if raw_permalink['clusterization_start_reason'] not in allowed_start_reasons:
                raw_permalink['clusterization_start_reason'] = 'common'

        table_data = {StoragePermalinkWL: raw_permalinks, StorageHotelWL: raw_hotels}
        local_storage.apply_data(table_data)

        for permalink in local_storage.permalinks_wl.values():

            for hotel in local_storage.get_permalink_wl_hotels(permalink):
                hotel.is_wl_approved = False

            is_processing_successful = Dispatcher.is_processing_successful(permalink)
            if is_processing_successful:
                permalink.wl_result = WLResult.SUCCESS
            else:
                permalink.wl_result = WLResult.FAILED
                continue

            for hotel in hotels.get(permalink.permalink, list()):
                storage_hotel = local_storage.hotels_wl.get(HotelWLKey.get_key(hotel))
                if storage_hotel is None:
                    storage_hotel = hotel
                    local_storage.add_hotel_wl(storage_hotel)
                storage_hotel.is_wl_approved = True

        self.mark_as_processed(StoragePermalinkWL, local_storage.permalinks_wl.values())

        LOG.info('Adding hotels to storage')
        self.add_fields_to_update(StorageHotelWL, ['grouping_key', 'partner_id', 'original_id', 'is_wl_approved'])
        self.update_storage_entity(local_storage.get_entity_data(StorageHotelWL), StorageHotelWL)

        LOG.info('Adding permalinks to storage')
        self.add_fields_to_update(StoragePermalinkWL, ['wl_result'])
        self.update_storage_entity(local_storage.get_entity_data(StoragePermalinkWL), StoragePermalinkWL)
