# -*- coding: utf-8 -*-

import logging

from yt.transfer_manager.client import TransferManager
from yt.wrapper import YtClient
import yt.wrapper as yt

from travel.hotels.content_manager.lib.processor import Processor

LOG = logging.getLogger(__name__)


class ProcessorWLExport(Processor):

    def run(self) -> None:
        hotels_wl_path = self.path_info.storage_hotels_wl_table
        direct_boy_export_path = self.options.direct_boy_export_path
        match_export_path = self.persistence_manager.join(self.options.export_path, 'matched_hotels')
        LOG.info(f'Exporting hotels_wl from {hotels_wl_path} to {direct_boy_export_path} and {match_export_path}')

        query = f'''
            INSERT INTO `{direct_boy_export_path}` WITH TRUNCATE
            SELECT
                permalink AS Permalink,
                partner_id AS PartnerId,
                original_id AS OriginalId,
            FROM `{hotels_wl_path}`
            WHERE
                partner_id IN ("PI_DOLPHIN", "PI_EXPEDIA", "PI_BRONEVIK") AND
                is_wl_approved == TRUE;

            INSERT INTO `{match_export_path}` WITH TRUNCATE
            SELECT
                permalink AS Permalink,
                partner_id AS PartnerId,
                original_id AS HotelCode,
            FROM `{hotels_wl_path}`
            WHERE
                partner_id IN ("PI_BNOVO", "PI_TRAVELLINE") AND
                is_wl_approved == TRUE;
        '''
        self.yql_client.run_query(query)

        permalinks_wl_path = self.path_info.storage_permalinks_wl_table
        wl_status_export_path = self.persistence_manager.join(self.options.export_path, 'wl_status')
        LOG.info(f'Exporting wl_status from {permalinks_wl_path} to {wl_status_export_path}')

        query = f'''
            $get_all_required_stages = (
                    $required_stages, $actualization_required_stages, $clusterization_required_stages
                ) -> (
                ListConcat(
                    ListFilter(
                        ListUniq(ListExtend(
                            String::SplitToList($required_stages, ","),
                            String::SplitToList($actualization_required_stages, ","),
                            String::SplitToList($clusterization_required_stages, ","),
                        )),
                        ($x) -> ($x != "")
                    ),
                    ","
                )
            );

            INSERT INTO `{wl_status_export_path}` WITH TRUNCATE
            SELECT
                permalink,
                $get_all_required_stages(
                    required_stages, actualization_required_stages, clusterization_required_stages
                ) AS all_required_stages,
                finished_stages,
                route,
                wl_result,
                priority,
                status_actualization,
                status_clusterization,
            FROM `{permalinks_wl_path}`
            WHERE required_stages != ""
        '''
        self.yql_client.run_query(query)

    def post_process(self):
        path = self.options.direct_boy_export_path
        transfer_manager = TransferManager(token=self.options.yt_token)
        for proxy in self.options.transfer_to:
            yt_client = YtClient(proxy=proxy, token=self.options.yt_token)
            dst_dir = yt.ypath_split(path)[0]
            if not yt_client.exists(dst_dir):
                yt_client.create('map_node', dst_dir, recursive=True)
            LOG.info(f'Transferring to {proxy}')
            transfer_manager.add_tasks(
                source_cluster=self.options.yt_proxy,
                source_pattern=path,
                destination_cluster=proxy,
                destination_pattern=path,
                enable_failed_tasks_restarting=True,
                include_files=True,
                sync=True,
            )

        LOG.info('All done')
