# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable, Type, Union
import logging
import random

from travel.hotels.content_manager.data_model.stage import WLGetHotelInfoHotelData, WLGetHotelInfoPermalinkData
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.processor import Processor


LOG = logging.getLogger(__name__)

BASE_LATITUDE = 56.835486

BASE_LONGITUDE = 60.590998


FIELDS_TO_SKIP = {
    'permalink',
    'partner_id',
    'original_id',
    'latitude',
    'longitude',
    'photo',
}


HotelInfoData = Union[WLGetHotelInfoHotelData, WLGetHotelInfoPermalinkData]


class MockWLGetHotelInfo(Processor):

    @staticmethod
    def update_fields(data: HotelInfoData, fields_to_update: Iterable[str]):
        permalink = data.permalink
        for field in fields_to_update:
            setattr(data, field, f'{field}_{permalink}')
        data.photo = [f'photo_{permalink}']
        data.latitude = BASE_LATITUDE + random.random()
        data.longitude = BASE_LONGITUDE + random.random()

    def update_info(self, info_cls: Type[HotelInfoData], table_name: str) -> None:
        input_table = self.persistence_manager.join(self.input_path, table_name)
        output_table = self.persistence_manager.join(self.output_path, table_name)

        output_data = list()
        schema = get_dc_yt_schema(info_cls)
        fields_to_update = set(schema.keys()) - FIELDS_TO_SKIP

        raw_data = self.persistence_manager.read(input_table)

        for input_value in raw_data:
            output_value: HotelInfoData = dc_from_dict(info_cls, input_value)
            self.update_fields(output_value, fields_to_update)
            output_data.append(dc_to_dict(output_value))

        LOG.info(f'Writing result to {output_table}')
        self.persistence_manager.write(output_table, output_data, schema)

    def run(self):
        LOG.info(f'Processing {self.input_path}')

        random.seed()

        self.update_info(WLGetHotelInfoHotelData, 'hotels')
        self.update_info(WLGetHotelInfoPermalinkData, 'permalinks')

        LOG.info('All done')
