# -*- coding: utf-8 -*-

import logging

from travel.hotels.content_manager.lib.processor import Processor


LOG = logging.getLogger(__name__)


class ProcessorWLGetHotelInfo(Processor):

    lang_priority = '''
            AsDict(
                AsTuple("RU", 0),
                AsTuple("EN", 1),
            );
    '''

    hotel_rubrics = '''
            AsDict(
                AsTuple(30655, "Общежитие"),
                AsTuple(30779, "Дом отдыха"),
                AsTuple(30781, "Санаторий"),
                AsTuple(30785, "Гостиница"),
                AsTuple(30788, "Кемпинг"),
                AsTuple(31309, "Отдых на ферме"),
                AsTuple(31632, "Хостел"),
                AsTuple(3501492236, "Апартаменты"),
                AsTuple(3501708107, "Жильё посуточно"),
            );
    '''

    def get_hotel_info(self, photos_table: str) -> None:
        hotels_input_table = self.persistence_manager.join(self.input_path, 'hotels')
        hotels_output_table = self.persistence_manager.join(self.output_path, 'hotels')

        query = f'''
            $lang_priority = {self.lang_priority}

            $hotel_rubrics = {self.hotel_rubrics}

            $get_item_by_lang_priority = ($list) -> (
                ListSort($list, ($x) -> ($lang_priority[$x.lang] ?? 100))[0]
            );

            $input_provider_names = (
                SELECT
                    input.permalink AS permalink,
                    input.partner_id AS partner_id,
                    partners.Code AS provider_id,
                    input.original_id AS original_id,
                FROM `{hotels_input_table}` AS input
                LEFT JOIN `home/travel/prod/config/partners` AS partners
                ON input.partner_id == partners.PartnerId
            );

            $input_provider_ids = (
                SELECT
                    input.permalink AS permalink,
                    input.partner_id AS partner_id,
                    input.original_id AS original_id,
                    provider.id AS provider_id,
                FROM $input_provider_names AS input
                LEFT JOIN `//home/altay/db/export/current-state/snapshot/provider` AS provider
                ON input.provider_id == provider.permalink
            );

            $minicards = (
                SELECT
                    input.permalink AS permalink,
                    input.partner_id AS partner_id,
                    input.original_id AS original_id,
                    input.provider_id AS provider_id,
                    minicards.data AS data,
                FROM $input_provider_ids as input
                LEFT JOIN `home/altay/db/clusterization/state-current/minicards` AS minicards
                ON
                    input.permalink == minicards.permalink AND
                    input.provider_id == minicards.data.provider_id AND
                    input.original_id == minicards.data.original_id
            );

            $get_photo = ($photos) -> (
                ListFlatMap($photos, ($x) -> ($x.link))
            );

            $photos_flat = (
                SELECT
                    permalink,
                    partner_id,
                    original_id,
                    photo,
                FROM (
                    SELECT
                        permalink,
                        partner_id,
                        original_id,
                        $get_photo(data.source.photos) AS photos,
                    FROM $minicards
                )
                FLATTEN LIST BY photos AS photo
            );

            $photos_mds_flat = (
                SELECT DISTINCT
                    photos.permalink AS permalink,
                    photos.partner_id AS partner_id,
                    photos.original_id AS original_id,
                    mds.url AS photo,
                FROM $photos_flat AS photos
                LEFT JOIN `{photos_table}` AS mds
                ON photos.photo == mds.original_url
                WHERE mds.url IS NOT NULL
            );

            $photos_mds = (
                SELECT
                    permalink,
                    partner_id,
                    original_id,
                    AGGREGATE_LIST(photo) AS photo,
                FROM $photos_mds_flat AS photos
                GROUP BY (permalink, partner_id, original_id)
            );

            $get_address = ($addresses) -> (
                $get_item_by_lang_priority($get_item_by_lang_priority($addresses).translations).one_line
            );

            $get_company_name = ($names) -> (
                $get_item_by_lang_priority($names).value
            );

            $get_country = ($addresses) -> (
                ListFilter(
                    $get_item_by_lang_priority($addresses).ad_hierarchy,
                    ($x) -> ($x.kind  == "COUNTRY")
                )[0].name ?? ""
            );

            $get_latitude = ($addresses) -> (
                $get_item_by_lang_priority($addresses).coordinates.lat
            );

            $get_longitude = ($addresses) -> (
                $get_item_by_lang_priority($addresses).coordinates.lon
            );

            $get_phone = ($phones) -> (
                String::JoinFromList(ListUniq(ListMap($phones, ($x) -> ($x.formatted))), ", ")
            );

            $get_category = ($rubrics) -> (
                String::JoinFromList(ListMap($rubrics, ($x) -> ($hotel_rubrics[$x.id] ?? "Неизвестно")), ", ")
            );

            $get_url = ($urls) -> (
                ListFilter($urls, ($x) -> ($x.type  == "MAIN"))[0].value ?? ""
            );

            INSERT INTO `{hotels_output_table}` WITH TRUNCATE
            SELECT
                minicards.permalink AS permalink,
                minicards.partner_id AS partner_id,
                minicards.original_id AS original_id,
                $get_address(minicards.data.company.addresses) AS address,
                $get_category(minicards.data.company.rubrics) AS category,
                $get_company_name(minicards.data.company.names) AS company_name,
                $get_country(minicards.data.company.addresses) AS country,
                $get_latitude(minicards.data.company.addresses) AS latitude,
                $get_longitude(minicards.data.company.addresses) AS longitude,
                $get_phone(minicards.data.company.phones) AS phone,
                photos.photo ?? [] AS photo,
                $get_url(minicards.data.source.urls) AS url,
            FROM $minicards AS minicards
            LEFT JOIN $photos_mds AS photos
            USING (permalink, partner_id, original_id)
        '''

        self.yql_client.run_query(query)

    def get_permalink_info(self, photos_table: str):
        permalinks_input_table = self.persistence_manager.join(self.input_path, 'permalinks')
        permalinks_output_table = self.persistence_manager.join(self.output_path, 'permalinks')

        query = f'''
            $lang_priority = {self.lang_priority}

            $hotel_rubrics = {self.hotel_rubrics}

            $permalinks_with_head = (
                SELECT
                    permalinks.permalink AS permalink,
                    CAST(map.cluster_permalink AS uint64) AS cluster_permalink,
                FROM `{permalinks_input_table}` AS permalinks
                LEFT JOIN `home/travel/prod/general/altay_mappings/latest/permalink_to_cluster_permalink` AS map
                USING(permalink)
            );

            $permalink_data = (
                SELECT
                    permalinks.permalink AS permalink,
                    companies.source_proto AS data,
                FROM $permalinks_with_head AS permalinks
                LEFT JOIN `home/altay/db/export/current-state/snapshot/company` AS companies
                ON permalinks.cluster_permalink == companies.permalink
            );

            $get_category = ($rubrics) -> (
                String::JoinFromList(ListMap($rubrics, ($x) -> ($hotel_rubrics[$x.rubric_id] ?? "Неизвестно")), ", ")
            );

            $get_company_name = ($names) -> (
                ListSort(
                    ListFilter($names, ($x) -> ($x.type  == "Main")),
                    ($x) -> ($lang_priority[$x.value.lang.locale] ?? 100)
                )[0].value.value
            );

            $get_country = ($address) -> (
                ListSort(
                    ListFilter($address.components, ($x) -> ($x.kind  == "Country")),
                    ($x) -> ($lang_priority[$x.name.lang.locale] ?? 100)
                )[0].name.value
            );

            $get_phone = ($phones) -> (
                String::JoinFromList(ListUniq(ListMap($phones, ($x) -> ($x.formatted))), ", ")
            );

            $get_photo_url = ($group, $name) -> (
                "https://avatars.mds.yandex.net/get-altay/" || Unwrap(CAST($group AS String)) || "/" || $name || "/orig"
            );

            $get_photo = ($photos) -> (
                ListFilter(
                    ListMap($photos, ($x) -> (
                        AsStruct(
                            $get_photo_url($x.value.group, $x.value.name) AS url,
                            $x.original_url AS original_url,
                            $x.status AS status,
                        )
                    )),
                    ($x) -> ($x.url IS NOT NULL AND $x.status == "Publish")
                )
            );

            $get_photo_urls = ($photo) -> (
                ListFlatMap($photo, ($x) -> ($x.url))
            );

            $get_url = ($urls) -> (
                ListFilter($urls, ($x) -> ($x.type  == "Main"))[0].value ?? ""
            );

            INSERT INTO `{photos_table}` WITH TRUNCATE
            SELECT
                photo.url AS url,
                photo.original_url AS original_url,
            FROM (
                SELECT
                    $get_photo(data.photos) AS photos,
                FROM $permalink_data
            )
            FLATTEN LIST BY photos AS photo;

            INSERT INTO `{permalinks_output_table}` WITH TRUNCATE
            SELECT
                permalink,
                data.address.formatted.value AS address,
                $get_category(data.rubrics) AS category,
                $get_company_name(data.names) AS company_name,
                $get_country(data.address) AS country,
                data.address.pos.point.lat AS latitude,
                data.address.pos.point.lon AS longitude,
                $get_phone(data.phones) AS phone,
                $get_photo_urls($get_photo(data.photos)) AS photo,
                $get_url(data.urls) AS url,
            FROM $permalink_data
        '''

        self.yql_client.run_query(query)

    def run(self) -> None:
        photos_table = self.persistence_manager.join(self.output_path, 'photos')

        LOG.info('Getting permalink info')
        self.get_permalink_info(photos_table)

        LOG.info('Getting hotel info')
        self.get_hotel_info(photos_table)

        LOG.info('Removing photos table')
        self.persistence_manager.delete(photos_table)

        LOG.info('All done')
