# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable, List
import logging

from travel.hotels.content_manager.data_model.stage import WLGetHotelInfoHotelData, WLGetHotelInfoPermalinkData
from travel.hotels.content_manager.data_model.storage import EntityWithStatus, StorageHotelWL, StoragePermalinkWL
from travel.hotels.content_manager.data_model.options import TriggerOptions
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.trigger import EntityGroupingKey, Producer, ThreadFilter, Trigger

LOG = logging.getLogger(__name__)


class ProducerWLGetHotelInfo(Producer):

    @staticmethod
    def get_hotels(storage_hotels: Iterable[StorageHotelWL]) -> List[WLGetHotelInfoHotelData]:
        hotels = list()
        for row in storage_hotels:
            hotel = WLGetHotelInfoHotelData(
                permalink=row.permalink,
                partner_id=row.partner_id,
                original_id=row.original_id,
            )
            hotels.append(hotel)
        return hotels

    @staticmethod
    def get_permalinks(storage_permalinks: Iterable[StoragePermalinkWL]) -> List[WLGetHotelInfoPermalinkData]:
        permalinks = list()
        for row in storage_permalinks:
            permalink = WLGetHotelInfoPermalinkData(
                permalink=row.permalink,
            )
            permalinks.append(permalink)
        return permalinks

    def prepare_data(
            self,
            trigger: Trigger,
            thread_filter: ThreadFilter,
            local_storage: Storage,
            path: str,
            entities: Iterable[EntityWithStatus],
            grouping_key: EntityGroupingKey,
            options: TriggerOptions,
    ):
        hotels_table = trigger.persistence_manager.join(path, 'hotels')
        hotels = self.get_hotels(entities)
        raw_hotels = (dc_to_dict(h) for h in hotels)
        hotels_schema = get_dc_yt_schema(WLGetHotelInfoHotelData)
        LOG.info(f'Writing hotels to {hotels_table}')
        trigger.persistence_manager.write(hotels_table, raw_hotels, hotels_schema)

        permalinks_table = trigger.persistence_manager.join(path, 'permalinks')
        permalink_ids = {e.permalink for e in entities}
        storage_permalinks = (local_storage.permalinks_wl[pid] for pid in permalink_ids)
        permalinks = self.get_permalinks(storage_permalinks)
        raw_permalinks = (dc_to_dict(p) for p in permalinks)
        permalinks_schema = get_dc_yt_schema(WLGetHotelInfoPermalinkData)
        LOG.info(f'Writing permalinks to {permalinks_table}')
        trigger.persistence_manager.write(permalinks_table, raw_permalinks, permalinks_schema)
