# -*- coding: utf-8 -*-
from __future__ import annotations

from dataclasses import dataclass
from typing import Iterable, List
import logging

from travel.hotels.content_manager.data_model.common import HotelInfo
from travel.hotels.content_manager.data_model.storage import StorageHotelWL, StoragePermalinkWL
from travel.hotels.content_manager.data_model.stage import WLGetHotelInfoHotelData, WLGetHotelInfoPermalinkData
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)


class UpdaterWLGetHotelInfo(Updater):

    @staticmethod
    def update_fields(dc_from: dataclass, dc_to: dataclass, fields_to_update: Iterable[str]):
        for field in fields_to_update:
            value = getattr(dc_from, field)
            setattr(dc_to, field, value)

    def get_hotels(
            self,
            output_path: str,
            fields_to_update: Iterable[str]
    ) -> List[StorageHotelWL]:
        hotels = list()

        hotels_table = self.persistence_manager.join(output_path, 'hotels')
        LOG.info(f'Reading hotels from {hotels_table}')
        raw_hotels_wl = self.persistence_manager.read(hotels_table)

        for raw_hotel_wl in raw_hotels_wl:
            data: WLGetHotelInfoHotelData = dc_from_dict(WLGetHotelInfoHotelData, raw_hotel_wl)
            LOG.info(data)
            hotel_wl = StorageHotelWL(
                permalink=data.permalink,
                partner_id=data.partner_id,
                original_id=data.original_id,
            )
            self.update_fields(data, hotel_wl, fields_to_update)
            hotels.append(hotel_wl)

        return hotels

    def get_permalinks(
            self,
            output_path: str,
            fields_to_update: Iterable[str]
    ) -> List[StoragePermalinkWL]:
        permalinks = list()

        permalinks_table = self.persistence_manager.join(output_path, 'permalinks')
        LOG.info(f'Reading permalinks from {permalinks_table}')
        raw_permalinks_wl = self.persistence_manager.read(permalinks_table)

        for raw_hotel_wl in raw_permalinks_wl:
            data: WLGetHotelInfoPermalinkData = dc_from_dict(WLGetHotelInfoPermalinkData, raw_hotel_wl)
            LOG.info(data)
            permalink_wl = StoragePermalinkWL(
                permalink=data.permalink,
            )
            self.update_fields(data, permalink_wl, fields_to_update)
            permalinks.append(permalink_wl)

        return permalinks

    def run(self, output_path: str, temp_dir: str) -> None:
        fields_to_update = get_dc_yt_schema(HotelInfo).keys()

        hotels = self.get_hotels(output_path, fields_to_update)
        self.mark_as_processed(StorageHotelWL, hotels)
        self.send_to_stage(StorageHotelWL, hotels, 'wl_match_hotels')

        permalinks = self.get_permalinks(output_path, fields_to_update)

        LOG.info('Adding hotels to storage')
        self.add_fields_to_update(StorageHotelWL, fields_to_update)
        raw_hotels = [dc_to_dict(h) for h in hotels]
        self.update_storage_entity(raw_hotels, StorageHotelWL)

        LOG.info('Adding permalinks to storage')
        self.add_fields_to_update(StoragePermalinkWL, fields_to_update)
        raw_permalinks = [dc_to_dict(p) for p in permalinks]
        self.update_storage_entity(raw_permalinks, StoragePermalinkWL)
