# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable
import logging
import math

from travel.hotels.content_manager.data_model.storage import StorageHotelWL, StoragePermalinkWL
from travel.hotels.content_manager.data_model.stage import (
    WLMatchHotelsData, WLMatchHotelsDataInput, WLMatchHotelsDataOutput
)
from travel.hotels.content_manager.data_model.storage import EntityWithStatus
from travel.hotels.content_manager.data_model.options import TriggerOptions
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.storage import HotelWLKey, Storage
from travel.hotels.content_manager.lib.trigger import EntityGroupingKey, Producer, ThreadFilter, Trigger


LOG = logging.getLogger(__name__)

EARTH_RADIUS = 6372795.


class ProducerWLMatchHotels(Producer):

    @staticmethod
    def get_distance(permalink_wl: StoragePermalinkWL, hotel_wl: StorageHotelWL) -> float:

        lat_1 = math.radians(permalink_wl.latitude)
        lon_1 = math.radians(permalink_wl.longitude)
        lat_2 = math.radians(hotel_wl.latitude)
        lon_2 = math.radians(hotel_wl.longitude)

        return EARTH_RADIUS * 2 * math.asin(math.sqrt(
            (math.sin((lat_2 - lat_1) / 2)) ** 2 +
            math.cos(lat_1) * math.cos(lat_2) * ((math.sin((lon_2 - lon_1) / 2)) ** 2)
        ))

    def get_task_row(self, permalink_wl: StoragePermalinkWL, hotel_wl: StorageHotelWL) -> WLMatchHotelsData:
        permalink_wl_photo = list(set(permalink_wl.photo) - set(hotel_wl.photo))

        match_hotels_input = WLMatchHotelsDataInput(
            permalink=str(permalink_wl.permalink),
            partner_id=hotel_wl.partner_id,
            original_id=hotel_wl.original_id,

            address_1=permalink_wl.address,
            category_1=permalink_wl.category,
            company_name_1=permalink_wl.company_name,
            country_1=permalink_wl.country,
            latitude_1=str(permalink_wl.latitude),
            longitude_1=str(permalink_wl.longitude),
            phone_1=permalink_wl.phone,
            photo_1=permalink_wl_photo,
            url_1=permalink_wl.url,

            address_2=hotel_wl.address,
            category_2=hotel_wl.category,
            company_name_2=hotel_wl.company_name,
            country_2=hotel_wl.country,
            latitude_2=str(hotel_wl.latitude),
            longitude_2=str(hotel_wl.longitude),
            phone_2=hotel_wl.phone,
            photo_2=hotel_wl.photo,
            url_2=hotel_wl.url,

            distance=self.get_distance(permalink_wl, hotel_wl),
        )

        return WLMatchHotelsData(
            input=match_hotels_input,
            output=WLMatchHotelsDataOutput(),
        )

    def prepare_data(
            self,
            trigger: Trigger,
            thread_filter: ThreadFilter,
            local_storage: Storage,
            path: str,
            entities: Iterable[EntityWithStatus],
            grouping_key: EntityGroupingKey,
            options: TriggerOptions,
    ):
        data = list()
        for entity in entities:
            hotel_wl = local_storage.hotels_wl[HotelWLKey(entity.permalink, entity.partner_id, entity.original_id)]
            permalink_wl = local_storage.permalinks_wl[hotel_wl.permalink]
            data.append(self.get_task_row(permalink_wl, hotel_wl))
        data_dict = (dc_to_dict(row) for row in data)

        hotels_table = trigger.persistence_manager.join(path, 'hotels')
        LOG.info(f'Writing task result to {hotels_table}')
        trigger.persistence_manager.write(hotels_table, data_dict, get_dc_yt_schema(WLMatchHotelsData))

        trigger.write_options(options, hotels_table, grouping_key)
