# -*- coding: utf-8 -*-
from __future__ import annotations

from enum import Enum
from typing import Iterable, List
import logging

from travel.hotels.content_manager.data_model.storage import StorageHotelWL
from travel.hotels.content_manager.data_model.stage import WLMatchHotelsData
from travel.hotels.content_manager.data_model.types import uint
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)


class MatchResult(Enum):
    MR_MATCH = 'double'
    MR_NOT_MATCH = 'no_double'


class UpdaterWLMatchHotels(Updater):

    def get_hotels(self, output_path: str) -> List[WLMatchHotelsData]:
        hotels = list()

        hotels_table = self.persistence_manager.join(output_path, 'hotels')
        LOG.info(f'Reading hotels from {hotels_table}')
        raw_hotels = self.persistence_manager.read(hotels_table)

        for raw_hotel in raw_hotels:
            hotel: WLMatchHotelsData = dc_from_dict(WLMatchHotelsData, raw_hotel)
            hotels.append(hotel)

        return hotels

    @staticmethod
    def get_storage_hotels(hotels: Iterable[WLMatchHotelsData]) -> List[StorageHotelWL]:
        storage_hotels = list()

        for hotel in hotels:
            storage_hotel = StorageHotelWL(
                permalink=uint(int(hotel.input.permalink)),
                partner_id=hotel.input.partner_id,
                original_id=hotel.input.original_id,
                is_wl_approved=MatchResult(hotel.output.result) == MatchResult.MR_MATCH,
            )
            storage_hotels.append(storage_hotel)

        return storage_hotels

    def run(self, output_path: str, temp_dir: str) -> None:
        hotels = self.get_hotels(output_path)
        LOG.info(f'Got {len(hotels)} result records')
        storage_hotels = self.get_storage_hotels(hotels)

        self.mark_as_processed(StorageHotelWL, storage_hotels)

        LOG.info('Adding hotels to storage')
        self.add_fields_to_update(StorageHotelWL, ['is_wl_approved'])
        raw_hotels = [dc_to_dict(h) for h in storage_hotels]
        self.update_storage_entity(raw_hotels, StorageHotelWL)
