# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable, List
import logging

from travel.hotels.content_manager.data_model.storage import StorageHotelWL, StoragePermalinkWL
from travel.hotels.content_manager.data_model.stage import WLNewHotelsHotelData
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)


class UpdaterWLNewHotels(Updater):

    def get_hotels(self, output_path: str) -> List[StorageHotelWL]:
        permalinks = list()

        permalinks_table = self.persistence_manager.join(output_path, 'hotels')
        LOG.info(f'Reading permalinks from {permalinks_table}')
        raw_permalinks = self.persistence_manager.read(permalinks_table)

        for raw_permalink in raw_permalinks:
            data: WLNewHotelsHotelData = dc_from_dict(WLNewHotelsHotelData, raw_permalink)
            LOG.info(data)
            permalink = StorageHotelWL(
                permalink=data.permalink,
                partner_id=data.partner_id,
                original_id=data.original_id,
                grouping_key=data.grouping_key,
            )
            permalinks.append(permalink)

        return permalinks

    @staticmethod
    def get_permalinks(hotels: Iterable[StorageHotelWL]) -> List[StoragePermalinkWL]:
        permalinks = list()
        permalink_ids = set()

        for hotel in hotels:
            if hotel.permalink in permalink_ids:
                continue
            permalink_wl = StoragePermalinkWL(
                permalink=hotel.permalink,
            )
            permalinks.append(permalink_wl)
            permalink_ids.add(hotel.permalink)

        return permalinks

    def run(self, output_path: str, temp_dir: str) -> None:
        hotels = self.get_hotels(output_path)
        permalinks = self.get_permalinks(hotels)
        self.send_to_stage(StorageHotelWL, hotels, 'wl_get_hotel_info')

        LOG.info('Adding hotels to storage')
        self.add_fields_to_update(StorageHotelWL, ['grouping_key', 'partner_id', 'original_id'])
        raw_hotels = [dc_to_dict(h) for h in hotels]
        self.update_storage_entity(raw_hotels, StorageHotelWL)

        LOG.info('Adding permalinks to storage')
        raw_permalinks = [dc_to_dict(p) for p in permalinks]
        self.update_storage_entity(raw_permalinks, StoragePermalinkWL)
