# -*- coding: utf-8 -*-
from __future__ import annotations

import dataclasses
import logging
from typing import Iterable, List

from travel.hotels.content_manager.data_model.storage import StoragePermalinkWL
from travel.hotels.content_manager.data_model.stage import WLStartData
from travel.hotels.content_manager.data_model.types import StageResult
from travel.hotels.content_manager.lib.common import dc_from_dict, str_from_set, str_to_set
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.persistence_manager import Condition
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)


class UpdaterWLStart(Updater):

    def get_output_data(self, output_path: str) -> List[WLStartData]:
        output_data = list()

        permalinks_table = self.persistence_manager.join(output_path, 'permalinks')
        LOG.info(f'Reading permalinks from {permalinks_table}')
        raw_data = self.persistence_manager.read(permalinks_table)

        for row in raw_data:
            item = dc_from_dict(WLStartData, row)
            output_data.append(item)

        return output_data

    def get_local_storage(self, output_data: Iterable[WLStartData]) -> Storage:
        permalink_ids = [int(p.permalink) for p in output_data]
        raw_permalinks = self.persistence_manager.request_select(
            self.path_info.storage_permalinks_wl_table,
            StoragePermalinkWL,
            match_conditions=[Condition('permalink', 'in', permalink_ids)],
        )

        # erase outdated clusterization_start_reason value as it will be replaced by new one
        for raw_permalink in raw_permalinks:
            raw_permalink['clusterization_start_reason'] = 'common'

        table_data = {StoragePermalinkWL: raw_permalinks}
        local_storage = Storage()
        local_storage.apply_data(table_data)
        return local_storage

    def update_permalinks(self, output_data: List[WLStartData], local_storage: Storage) -> None:
        processed_permalinks = list()
        for item in output_data:
            LOG.info(item)

            storage_permalink = local_storage.permalinks_wl.get(item.permalink)
            if storage_permalink is None:
                storage_permalink = StoragePermalinkWL(permalink=item.permalink)
                local_storage.add_permalink_wl(storage_permalink)

            self.set_permalink_value(item, storage_permalink)

            processed_permalinks.append(storage_permalink)
            LOG.info(storage_permalink)

        self.dispatch_entities(StoragePermalinkWL, processed_permalinks, 'wl_clusterized_hotels')

    def set_permalink_value(self, data: WLStartData, permalink: StoragePermalinkWL) -> None:
        is_entity_processed = self.dispatcher.is_entity_processed(permalink)
        if is_entity_processed:
            permalink.actualization_result = StageResult.UNKNOWN
            permalink.call_center_result = StageResult.UNKNOWN
            permalink.clusterization_result = StageResult.UNKNOWN
            permalink.route = ''
            permalink.finished_stages = ''
            permalink.checked_attributes = ''
            permalink.comments = ''
        for field in dataclasses.fields(data):
            if field.name == 'permalink':
                continue
            try:
                self.set_field_value(data, permalink, field, is_entity_processed)
            except Exception as e:
                LOG.error(f'Failed to set field value {field}')
                raise e

    @staticmethod
    def set_field_value(
        data: WLStartData,
        permalink: StoragePermalinkWL,
        field: dataclasses.field,
        replace_all: bool,
    ) -> None:
        metadata = dict(**field.metadata)
        value = getattr(data, field.name)
        item_type = metadata.get('item_type')
        converted_value = None
        if item_type is not None:
            converted_value = str_to_set(value, lambda x: item_type(x))
        replace = metadata.get('replace') or replace_all
        if replace:
            value = value or metadata.get('value')
            if value is None:
                raise Exception(f'No value for field {field}')
            setattr(permalink, field.name, value)
            return
        if value is None:
            return
        permalink_value = getattr(permalink, field.name)
        merger = metadata.get('merger')
        if merger is not None:
            setattr(permalink, field.name, merger(permalink_value, value))
            return
        if item_type is None:
            setattr(permalink, field.name, value)
        permalink_value = str_to_set(permalink_value, lambda x: item_type(x))
        value = str_from_set(permalink_value | converted_value, lambda x: x.value)
        setattr(permalink, field.name, value)

    def run(self, output_path: str, temp_dir: str) -> None:
        output_data = self.get_output_data(output_path)
        LOG.info(f'Got {len(output_data)} result records')

        local_storage = self.get_local_storage(output_data)
        self.update_permalinks(output_data, local_storage)

        LOG.info('Adding permalinks to storage')
        fields_to_update = [
            'grouping_key',
            'priority',
            'required_stages',
            'finished_stages',
            'route',
            'actualization_start_reason',
            'clusterization_start_reason',
            'hotel_name',
            'assignee_skill',
            'requirements',
            'required_attributes',
            'actualization_iteration',
            'actualization_required_stages',
            'call_center_required_stages',
            'clusterization_required_stages',
            'clusterization_iteration',
            'actualization_result',
            'call_center_result',
            'clusterization_result',
        ]
        self.add_fields_to_update(StoragePermalinkWL, fields_to_update)
        self.update_storage_entity(local_storage.get_entity_data(StoragePermalinkWL), StoragePermalinkWL)
