# -*- coding: utf-8 -*-
from __future__ import annotations

from datetime import datetime
from uuid import uuid4
import logging
import random

from travel.hotels.content_manager.data_model.stage import YangMappingsData
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.mocks_common import normalized_room_name
from travel.hotels.content_manager.lib.processor import Processor


LOG = logging.getLogger(__name__)

NEW_PERMAROOM_NAME = 'new_permaroom'

MAPPING_TO_HIDE_NAME = 'unknown'


class MockMonkeyMappings(Processor):

    @staticmethod
    def get_result(offer: YangMappingsData) -> str:
        name_mapping = dict()
        for permaroom in offer.input.permarooms:
            permaroom_name = permaroom.permaroom_name
            name_mapping[normalized_room_name(permaroom_name)] = permaroom
        normalized_offer_room_name = normalized_room_name(offer.input.offer_room_name)
        permaroom = name_mapping.get(normalized_offer_room_name)
        if permaroom is None:
            return random.choice([NEW_PERMAROOM_NAME, MAPPING_TO_HIDE_NAME])
        return permaroom.permaroom_id

    @staticmethod
    def set_info(task: YangMappingsData, start_ts: int) -> None:
        pool = random.choice([1, 2])
        info = task.info
        info.assignment_id = str(uuid4())
        info.create_ts = start_ts
        info.pool_id = str(pool)
        info.reward = float(pool)
        info.status = 'ACCEPTED'
        info.submit_ts = start_ts + random.randint(300, 900)
        info.worker_id = str(random.randint(0, 10))

    def run(self):
        LOG.info(f'Processing {self.input_path}')
        random.seed()

        start_ts = int(datetime.utcnow().timestamp())

        input_table = self.persistence_manager.join(self.input_path, 'yang_mappings')
        output_table = self.persistence_manager.join(self.output_path, 'yang_mappings')
        raw_input = self.persistence_manager.read(input_table)
        output_data = list()
        for row in raw_input:
            mappings_data: YangMappingsData = dc_from_dict(YangMappingsData, row)
            mappings_data.output.result = self.get_result(mappings_data)
            mappings_data.output.offer_comment = 'made by monkey'
            self.set_info(mappings_data, start_ts)
            output_data.append(dc_to_dict(mappings_data))
        LOG.debug(f'Got {len(output_data)} mappings')

        LOG.debug(f'Write result to {output_table}')
        self.persistence_manager.write(output_table, output_data, get_dc_yt_schema(YangMappingsData))

        LOG.info('All done')
