# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable
import logging

from travel.hotels.content_manager.data_model.storage import StorageMapping, StoragePermalink
from travel.hotels.content_manager.data_model.storage import StoragePermaroom
from travel.hotels.content_manager.data_model.stage import AvailablePermaroom, MappingTaskInfo, YangMappingsData
from travel.hotels.content_manager.data_model.stage import YangMappingsInputData, YangMappingsOutputData
from travel.hotels.content_manager.data_model.storage import EntityWithStatus
from travel.hotels.content_manager.data_model.options import TriggerOptions
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.trigger import EntityGroupingKey, Producer, ThreadFilter, Trigger


LOG = logging.getLogger(__name__)

ENTITIES_TO_READ = [
    StoragePermalink,
    StoragePermaroom,
    StorageMapping,
]


class ProducerYangMappings(Producer):

    @staticmethod
    def get_task_row(local_storage: Storage, permalink: StoragePermalink, offer: StorageMapping) -> YangMappingsData:

        permarooms = list()
        for permaroom in local_storage.get_permalink_permarooms(permalink).values():
            if permaroom.name == 'hidden':
                continue
            mp = AvailablePermaroom(
                permaroom_id=str(permaroom.id),
                permaroom_name=permaroom.name,
                permaroom_comment=permaroom.comment,
                alternative_names=permaroom.alternative_names,
            )
            permarooms.append(mp)

        input_data = YangMappingsInputData(
            permalink=str(offer.permalink),
            permarooms=permarooms,
            operator_id=offer.operator_id,
            orig_hotel_id=offer.orig_hotel_id,
            mapping_key=offer.mapping_key,
            offer_room_name=offer.orig_room_name,
            count=str(offer.count),
            prices_per_night=offer.prices_per_night,
            requests_range=offer.requests_range,
            price=str(offer.price),
            partner_offer_url=offer.url,
            altay_url=f'https://altay.yandex-team.ru/cards/perm/{offer.permalink}',
            offer_params=offer.params,
        )

        row = YangMappingsData(
            input=input_data,
            output=YangMappingsOutputData(),
            info=MappingTaskInfo(),
        )

        return row

    def prepare_data(
            self,
            trigger: Trigger,
            thread_filter: ThreadFilter,
            local_storage: Storage,
            path: str,
            entities: Iterable[EntityWithStatus],
            grouping_key: EntityGroupingKey,
            options: TriggerOptions,
    ):
        data = list()
        permalink_ids = {e.permalink for e in entities}
        for permalink_id in permalink_ids:
            permalink = local_storage.permalinks[permalink_id]
            for offer in local_storage.get_permalink_other_mappings(permalink).values():
                data.append(self.get_task_row(local_storage, permalink, offer))
        data_dict = (dc_to_dict(row) for row in data)

        yang_mappings_table = trigger.persistence_manager.join(path, 'yang_mappings')
        LOG.info(f'Writing task result to {yang_mappings_table}')
        trigger.persistence_manager.write(yang_mappings_table, data_dict, get_dc_yt_schema(YangMappingsData))

        trigger.write_options(options, yang_mappings_table, grouping_key)
