# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable, List
import logging

from travel.hotels.content_manager.data_model.storage import StorageMapping, StoragePermalink
from travel.hotels.content_manager.data_model.storage import StoragePermaroom
from travel.hotels.content_manager.data_model.stage import MappingTaskInfo, YangMappingsData
from travel.hotels.content_manager.data_model.types import uint
from travel.hotels.content_manager.lib.common import dc_from_dict
from travel.hotels.content_manager.lib.persistence_manager import Condition
from travel.hotels.content_manager.lib.storage import MappingKey, Storage
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)

ENTITIES_TO_PROCESS = [
    StoragePermalink,
    StoragePermaroom,
    StorageMapping,
]


class UpdaterYangMappings(Updater):

    def get_mappings(self, output_path: str) -> List[YangMappingsData]:
        mappings = list()
        path = self.persistence_manager.join(output_path, 'yang_mappings')
        for row in self.persistence_manager.read(path):
            mapping = dc_from_dict(YangMappingsData, row)
            mappings.append(mapping)
        return mappings

    def update_local_storage(self, local_storage: Storage, processed_permalinks: List[int]) -> None:
        LOG.info('Getting storage info on new mappings')

        table_data = dict()
        for entity_cls in ENTITIES_TO_PROCESS:
            src_path = self.entity_to_path[entity_cls]
            permalink_field = 'id' if entity_cls is StoragePermalink else 'permalink'

            LOG.info(f'Getting {entity_cls} from {src_path}')
            entities = self.persistence_manager.request_select(
                src_path=src_path,
                dc=entity_cls,
                match_conditions=[Condition(permalink_field, 'in', processed_permalinks)],
            )
            table_data[entity_cls] = entities

        local_storage.apply_data(table_data)

    def update_mappings(self, local_storage: Storage, mappings_data: List[YangMappingsData]) -> None:
        permalinks_to_edit = set()
        processed_mappings = list()

        for mapping_info in mappings_data:
            LOG.info(f'Processing {mapping_info}')

            mapping = StorageMapping(
                permalink=uint(int(mapping_info.input.permalink)),
                operator_id=mapping_info.input.operator_id,
                orig_hotel_id=mapping_info.input.orig_hotel_id,
                mapping_key=mapping_info.input.mapping_key,
            )

            permalink = local_storage.permalinks[uint(int(mapping_info.input.permalink))]
            mapping_key = MappingKey.get_key(mapping)

            full_mapping = local_storage.mappings[mapping_key]
            full_mapping.comment = mapping_info.output.offer_comment
            processed_mappings.append(full_mapping)

            if mapping_info.output.result == 'new_permaroom':
                need_new_permaroom_counter = full_mapping.counters.get('need_new_permaroom', 0)
                need_new_permaroom_counter += 1
                full_mapping.counters['need_new_permaroom'] = need_new_permaroom_counter
                if need_new_permaroom_counter >= 3:
                    full_mapping.is_banned = True
                    LOG.info(f'Mapping was banned {mapping}')
                    continue

                LOG.info('Mapping need new permaroom')
                full_mapping.need_new_permaroom = True
                permalinks_to_edit.add(permalink.id)
                continue

            if mapping_info.output.result == 'unknown':
                LOG.info('Mapping is hidden')
                full_mapping.is_hidden = True
                continue

            permaroom_id = int(mapping_info.output.result)
            permaroom = local_storage.permarooms.get(permaroom_id)
            if permaroom is None:
                raise KeyError(f'No suitable permaroom for {mapping_info}')
            full_mapping.need_new_permaroom = False
            local_storage.delete_other_mapping(mapping_key)
            full_mapping.permaroom_id = permaroom_id
            local_storage.add_mapping(permalink, full_mapping)

        self.mark_as_processed(StorageMapping, processed_mappings)
        self.send_to_stage(StoragePermalink, (local_storage.permalinks[p] for p in permalinks_to_edit), 'yang_rooms')

        mapping_fields_to_update = [
            'comment',
            'counters',
            'is_banned',
            'is_hidden',
            'need_new_permaroom',
            'permaroom_id',
        ]
        self.add_fields_to_update(StorageMapping, mapping_fields_to_update)

    def get_task_logs(self, mappings: Iterable[YangMappingsData]) -> List[MappingTaskInfo]:
        logs = list()
        for mapping in mappings:
            task_info = mapping.info
            task_info.stage = self.stage_name

            task_info.permalink = int(mapping.input.permalink)
            task_info.operator_id = mapping.input.operator_id
            task_info.orig_hotel_id = mapping.input.orig_hotel_id
            task_info.mapping_key = mapping.input.mapping_key

            task_info.reward = float(task_info.reward)
            logs.append(task_info)
        return logs

    def update_storage(self, local_storage: Storage) -> None:
        LOG.info('Update storage permalinks status')
        self.update_storage_entity(local_storage.get_entity_data(StoragePermalink), StoragePermalink)

        LOG.info('Update storage permarooms')
        self.update_storage_entity(local_storage.get_entity_data(StoragePermaroom), StoragePermaroom)

        LOG.info('Update storage mappings')
        self.update_storage_entity(local_storage.get_entity_data(StorageMapping), StorageMapping)

    def run(self, output_path: str, temp_dir: str) -> None:
        mappings_data = self.get_mappings(output_path)
        LOG.debug(f'Got mappings data on {len(mappings_data)} permalinks')

        self.update_task_logs(self.get_task_logs(mappings_data), MappingTaskInfo, 'mappings')

        processed_permalinks = [int(m.input.permalink) for m in mappings_data]
        local_storage = Storage()
        self.update_local_storage(local_storage, processed_permalinks)

        self.update_mappings(local_storage, mappings_data)
        self.update_storage(local_storage)
