# -*- coding: utf-8 -*-
from __future__ import annotations

from datetime import datetime
from typing import Any, Dict, Iterable, List, Optional
from uuid import uuid4
import logging
import random

from travel.hotels.content_manager.data_model.stage import AvailablePermaroom, PermalinkOffer, YangRoomsData
from travel.hotels.content_manager.lib.common import dc_from_dict, dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.mocks_common import normalized_room_name
from travel.hotels.content_manager.lib.processor import Processor


LOG = logging.getLogger(__name__)

EDITED_PERMAROOM_NAME_PREFIX = 'edited_'


class MockMonkeyRooms(Processor):

    @staticmethod
    def get_available_permarooms(permarooms: List[AvailablePermaroom]) -> Dict[str, AvailablePermaroom]:
        result = dict()
        for p in permarooms:
            permaroom = AvailablePermaroom(
                permaroom_id=p.permaroom_id,
                permaroom_name=p.permaroom_name,
                alternative_names=p.alternative_names,
                permaroom_comment=p.permaroom_comment,
            )
            result[p.permaroom_name] = permaroom
        return result

    @staticmethod
    def get_offer_permarooms(offers: List[PermalinkOffer]) -> Dict[str, AvailablePermaroom]:
        permarooms = dict()
        permaroom_names = set(normalized_room_name(o.orig_room_name) for o in offers)
        for permaroom_name in permaroom_names:
            permaroom = AvailablePermaroom(
                permaroom_id=None,
                permaroom_name=permaroom_name,
                permaroom_comment='made by monkey',
            )
            permarooms[permaroom_name] = permaroom
        return permarooms

    @staticmethod
    def get_edited_permaroom(permarooms: Iterable[AvailablePermaroom]) -> Optional[AvailablePermaroom]:
        for permaroom in permarooms:
            if permaroom.permaroom_name.startswith(EDITED_PERMAROOM_NAME_PREFIX):
                return permaroom

    def get_permalink_rooms_data(self, raw_data: Dict[str, Any]) -> YangRoomsData:
        rooms_data: YangRoomsData = dc_from_dict(YangRoomsData, raw_data)

        available_permaroom_by_name = {p.permaroom_name: p for p in rooms_data.input.available_permarooms}
        new_permarooms = self.get_offer_permarooms(rooms_data.input.permalink_offers)
        new_permarooms.update(available_permaroom_by_name)
        new_permarooms = list(new_permarooms.values())

        edited_permaroom = self.get_edited_permaroom(new_permarooms)
        if edited_permaroom:
            edited_permaroom.permaroom_name = edited_permaroom.permaroom_name[len(EDITED_PERMAROOM_NAME_PREFIX):]
        elif rooms_data.input.available_permarooms:
            permaroom_to_edit = random.choice(rooms_data.input.available_permarooms)
            permaroom_to_edit.permaroom_name = EDITED_PERMAROOM_NAME_PREFIX + permaroom_to_edit.permaroom_name

        permaroom_to_delete = random.choice(new_permarooms)
        new_permarooms.remove(permaroom_to_delete)

        rooms_data.output.result_permarooms = new_permarooms
        return rooms_data

    @staticmethod
    def set_info(task: YangRoomsData, start_ts: int) -> None:
        pool = random.choice([1, 2])
        info = task.info
        info.assignment_id = str(uuid4())
        info.create_ts = start_ts
        info.pool_id = str(pool)
        info.reward = float(pool)
        info.status = 'ACCEPTED'
        info.submit_ts = start_ts + random.randint(300, 900)
        info.worker_id = str(random.randint(0, 10))

    def run(self):
        LOG.info(f'Processing {self.input_path}')
        random.seed()

        start_ts = int(datetime.utcnow().timestamp())

        input_table = self.persistence_manager.join(self.input_path, 'yang_rooms')
        output_table = self.persistence_manager.join(self.output_path, 'yang_rooms')

        raw_input = self.persistence_manager.read(input_table)

        output_data = list()
        for row in raw_input:
            rooms_data = self.get_permalink_rooms_data(row)
            self.set_info(rooms_data, start_ts)
            output_data.append(dc_to_dict(rooms_data))
        LOG.info(f'Processed {len(output_data)} permalinks')

        LOG.info(f'Write result to {output_table}')
        self.persistence_manager.write(output_table, output_data, get_dc_yt_schema(YangRoomsData))

        LOG.info('All done')
