# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable, List
import logging

from travel.hotels.content_manager.data_model.options import TriggerOptions
from travel.hotels.content_manager.data_model.storage import EntityWithStatus, StoragePermalink
from travel.hotels.content_manager.data_model.stage import (
    AvailablePermaroom, OfferInfo, PermalinkOffer, PermalinkTaskInfo, YangRoomsInputData, YangRoomsOutputData,
    YangRoomsData
)
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.trigger import EntityGroupingKey, Producer, ThreadFilter, Trigger


LOG = logging.getLogger(__name__)


class ProducerYangRooms(Producer):

    @staticmethod
    def get_all_offers(local_storage: Storage, permalink: StoragePermalink):
        for offer in local_storage.get_permalink_mappings(permalink).values():
            yield offer

    def get_permalink_offers(self, local_storage: Storage, permalink: StoragePermalink) -> List[PermalinkOffer]:
        offers = list()

        for offer in self.get_all_offers(local_storage, permalink):
            permalink_offer = PermalinkOffer(
                operator_id=offer.operator_id,
                orig_hotel_id=offer.orig_hotel_id,
                mapping_key=offer.mapping_key,
                orig_room_name=offer.orig_room_name,
                _price=str(offer.price),
                _prices_per_night=offer.prices_per_night,
                _count=str(offer.count),
                _params=offer.params,
                _partner_url=offer.url,
                _requests_range=offer.requests_range,
            )
            offers.append(permalink_offer)

        return offers

    @staticmethod
    def get_offer_info(local_storage: Storage, permalink: StoragePermalink) -> List[OfferInfo]:
        offers = list()

        for offer in local_storage.get_permalink_other_mappings(permalink).values():
            if not offer.need_new_permaroom:
                continue
            offer_info = OfferInfo(
                offer_room_name=offer.orig_room_name,
                partner_offer_url=offer.url,
                offer_comment=offer.comment,
            )
            offers.append(offer_info)

        return offers

    @staticmethod
    def get_available_permarooms(local_storage: Storage, permalink: StoragePermalink) -> List[AvailablePermaroom]:
        permarooms = list()

        for permaroom in local_storage.get_permalink_permarooms(permalink).values():
            if permaroom.is_deleted:
                continue
            available_permaroom = AvailablePermaroom(
                permaroom_id=str(permaroom.id),
                permaroom_name=permaroom.name,
                alternative_names=permaroom.alternative_names,
                permaroom_comment=permaroom.comment,
            )
            permarooms.append(available_permaroom)

        return permarooms

    @staticmethod
    def separate_permalinks(
        local_storage: Storage,
        permalinks: Iterable[StoragePermalink],
    ) -> (List[StoragePermalink], List[StoragePermalink]):
        new_permalinks = list()
        permalinks_to_edit = list()

        for permalink in permalinks:
            if local_storage.permalink_has_permarooms(permalink):
                permalinks_to_edit.append(permalink)
            else:
                new_permalinks.append(permalink)

        return new_permalinks, permalinks_to_edit

    def prepare_data(
            self,
            trigger: Trigger,
            thread_filter: ThreadFilter,
            local_storage: Storage,
            path: str,
            entities: Iterable[EntityWithStatus],
            grouping_key: EntityGroupingKey,
            options: TriggerOptions,
    ):
        LOG.info('Preparing trigger data')
        task_data = list()

        for permalink in entities:
            LOG.debug(permalink)
            permalink = local_storage.permalinks[permalink.id]
            urls = list()
            for url_key in local_storage.get_permalink_urls(permalink):
                url = local_storage.urls[url_key]
                urls.extend(url)
            partner_urls = [{u.provider: u.url} for u in urls]
            alt_hotel_url = urls[0].url if urls else f'https://yandex.ru/search/?text={permalink.id}'
            hotel_url = permalink.hotel_url or alt_hotel_url
            if not hotel_url:
                raise RuntimeError(f'No hotel_url for {permalink}')

            permalink_offers = self.get_permalink_offers(local_storage, permalink)
            if not permalink_offers:
                raise RuntimeError(f'{permalink} has no offers')

            yang_permalink = YangRoomsData(
                YangRoomsInputData(
                    permalink=str(permalink.id),
                    altay_url=f'https://altay.yandex-team.ru/cards/perm/{permalink.id}',
                    hotel_url=hotel_url,
                    partner_urls=partner_urls,
                    permalink_offers=permalink_offers,
                    # used for permalink structure editing
                    offer_info=self.get_offer_info(local_storage, permalink),
                    available_permarooms=self.get_available_permarooms(local_storage, permalink),
                ),
                YangRoomsOutputData(),
                info=PermalinkTaskInfo(),
            )

            task_data.append(dc_to_dict(yang_permalink))

        yang_task_table = trigger.persistence_manager.join(path, 'yang_rooms')
        LOG.info(f'Writing result to {yang_task_table}')
        trigger.persistence_manager.write(yang_task_table, task_data, get_dc_yt_schema(YangRoomsData))

        trigger.write_options(options, yang_task_table, grouping_key)


def filter_create_structure(permalink: StoragePermalink, storage: Storage) -> bool:
    return not storage.permalink_has_permarooms(permalink)


def filter_edit_structure(permalink: StoragePermalink, storage: Storage) -> bool:
    return storage.permalink_has_permarooms(permalink)


FILTERS_YANG_ROOMS = [
    ThreadFilter('create_structure', 'create_structure', filter_create_structure),
    ThreadFilter('edit_structure', 'edit_structure', filter_edit_structure),
]
