# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Iterable, List
from dataclasses import asdict
import logging

from travel.hotels.content_manager.data_model.storage import PERMAROOM_ID_OFFSET
from travel.hotels.content_manager.data_model.storage import StorageMapping, StoragePermalink
from travel.hotels.content_manager.data_model.storage import StoragePermaroom
from travel.hotels.content_manager.data_model.stage import AvailablePermaroom, PermalinkTaskInfo, YangRoomsData
from travel.hotels.content_manager.data_model.types import uint
from travel.hotels.content_manager.lib.common import dc_from_dict
from travel.hotels.content_manager.lib.persistence_manager import Condition
from travel.hotels.content_manager.lib.storage import Storage
from travel.hotels.content_manager.lib.updater import Updater


LOG = logging.getLogger(__name__)

ENTITIES_TO_PROCESS = [
    StoragePermalink,
    StoragePermaroom,
    StorageMapping,
]


class UpdaterYangRooms(Updater):

    def __init__(self, **kwargs):
        super(UpdaterYangRooms, self).__init__(**kwargs)

        permarooms_count = self.persistence_manager.row_count(self.path_info.storage_permarooms_table)
        self.permaroom_id: int = PERMAROOM_ID_OFFSET + permarooms_count

    def get_rooms_data(self, output_path: str) -> List[YangRoomsData]:
        path = self.persistence_manager.join(output_path, 'yang_rooms')
        results = list()
        for row in self.persistence_manager.read(path):
            result = dc_from_dict(YangRoomsData, row)
            results.append(result)
        return results

    @staticmethod
    def get_permalinks(rooms_data: List[YangRoomsData]) -> List[int]:
        permalinks = {int(item.input.permalink) for item in rooms_data}
        return list(permalinks)

    def get_task_logs(self, hotels: Iterable[YangRoomsData]) -> List[PermalinkTaskInfo]:
        logs = list()
        for result in hotels:
            task_info = result.info
            task_info.stage = self.stage_name
            task_info.permalink = int(result.input.permalink)
            task_info.reward = float(task_info.reward)
            logs.append(task_info)
        return logs

    def update_local_storage(self, local_storage: Storage, processed_permalinks: List[int]) -> None:
        LOG.info('Getting storage info on processed permalinks')

        table_data = dict()
        for entity_cls in ENTITIES_TO_PROCESS:
            src_path = self.entity_to_path[entity_cls]
            permalink_field = 'id' if entity_cls is StoragePermalink else 'permalink'

            LOG.info(f'Getting {entity_cls} from {src_path}')
            entities = self.persistence_manager.request_select(
                src_path=src_path,
                dc=entity_cls,
                match_conditions=[Condition(permalink_field, 'in', processed_permalinks)],
            )
            table_data[entity_cls] = entities

        local_storage.apply_data(table_data)

    def update_rooms(self, storage: Storage, rooms_data: List[YangRoomsData]) -> None:
        for row in rooms_data:
            self.update_permalink_rooms(storage, row)

    def add_permaroom(
        self,
        local_storage: Storage,
        permalink: StoragePermalink,
        room: AvailablePermaroom,
    ) -> StoragePermaroom:
        permaroom = StoragePermaroom()
        permaroom.id = self.permaroom_id
        self.permaroom_id += 1
        permaroom.name = room.permaroom_name
        permaroom.permalink = permalink.id
        permaroom.alternative_names = room.alternative_names
        permaroom.comment = room.permaroom_comment
        local_storage.add_permaroom(permalink, permaroom)
        return permaroom

    def update_permalink_rooms(self, storage: Storage, row: YangRoomsData) -> None:
        permalink = storage.permalinks[uint(int(row.input.permalink))]
        LOG.info(f'Checking rooms for {permalink}')

        current_ids = set(storage.get_permalink_permarooms(permalink))
        output_ids = set()
        all_names = set()

        for room in row.output.result_permarooms:
            LOG.info(f'Checking {room}')
            name = room.permaroom_name

            if name in all_names:
                # TODO: some sensible action
                LOG.error(f'Room name duplicates for "{name}"')
                continue

            if room.permaroom_id is not None:
                output_ids.add(int(room.permaroom_id))
            all_names.add(name)

            if room.permaroom_id is None:
                LOG.info('New permaroom')
                self.add_permaroom(storage, permalink, room)
                continue

            permaroom_id = int(room.permaroom_id)
            permaroom = storage.permarooms.get(permaroom_id)

            if permaroom is None:
                raise RuntimeError(f'Unknown permaroom_id: {permaroom_id}')

            current_name = permaroom.name
            if current_name != name:
                LOG.info(f'Permaroom renamed from "{current_name}" to "{name}"')

            permaroom.name = name
            permaroom.alternative_names = room.alternative_names
            permaroom.comment = room.permaroom_comment

        permaroom_ids_to_delete = current_ids - output_ids
        LOG.info(f'Permarooms to delete: {permaroom_ids_to_delete}')
        for permaroom_id in permaroom_ids_to_delete:
            storage.delete_permaroom(permalink, permaroom_id)

    def update_status(self, local_storage: Storage) -> None:
        LOG.info('Updating storage status')
        self.mark_as_processed(StoragePermalink, (p for p in local_storage.permalinks.values()))

        mappings_to_send = list()
        for permalink in local_storage.permalinks.values():
            for mapping in local_storage.get_permalink_other_mappings(permalink).values():
                if mapping.is_hidden or mapping.is_banned or mapping.orig_room_name == '':
                    continue
                mappings_to_send.append(mapping)
        self.send_to_stage(StorageMapping, mappings_to_send, 'offer_prioritization')
        self.add_fields_to_update(StorageMapping, ['permaroom_id'])

    def update_storage(self, local_storage: Storage) -> None:
        LOG.info('Update storage permalinks status')
        self.update_storage_entity(local_storage.get_entity_data(StoragePermalink), StoragePermalink)

        LOG.info('Writing new permarooms to storage')
        self.add_fields_to_update(StoragePermaroom, asdict(StoragePermaroom()).keys())
        self.update_storage_entity(local_storage.get_entity_data(StoragePermaroom), StoragePermaroom)

        LOG.info('Update storage mappings')
        self.update_storage_entity(local_storage.get_entity_data(StorageMapping), StorageMapping)

    def run(self, output_path: str, temp_dir: str) -> None:
        rooms_data = self.get_rooms_data(output_path)
        LOG.debug(f'Got rooms data on {len(rooms_data)} permalinks')
        processed_permalinks = self.get_permalinks(rooms_data)
        self.update_task_logs(self.get_task_logs(rooms_data), PermalinkTaskInfo, 'permalinks')

        local_storage = Storage()
        self.update_local_storage(local_storage, processed_permalinks)

        self.update_rooms(local_storage, rooms_data)
        self.update_status(local_storage)
        self.update_storage(local_storage)
