# -*- coding: utf-8 -*-
from __future__ import annotations

from typing import Any, Dict, Generator, List, Set
import logging
import random

from travel.hotels.content_manager.data_model.storage import StageStatus, StorageMapping, StorageUrl
from travel.hotels.content_manager.data_model.types import uint
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.storage import MappingKey, UrlKey
from travel.hotels.content_manager.lib.processor import Processor

LOG = logging.getLogger(__name__)

NEW_MAPPINGS_MIN = 5
NEW_MAPPINGS_MAX = 20

PERMAROOM_INDEX_MAX = 25

OFFER_COUNT_MAX = 10_000

ORIG_HOTEL_ID_MAX = 1_000_000_000

PRICE_MAX = 1_000_000


class MockPartnerOffers(Processor):

    def get_known_mappings(self) -> Set[tuple]:
        result = set()
        for row in self.persistence_manager.read(self.path_info.storage_mappings_table):
            mapping = StorageMapping(**row)
            value = MappingKey.get_key(mapping)
            if value in result:
                raise RuntimeError(f'Value duplicates: {value}')
            result.add(value)
        return result

    def get_known_urls(self) -> Set[tuple]:
        result = set()
        for row in self.persistence_manager.read(self.path_info.storage_urls_table):
            url = StorageUrl(**row)
            value = (url.permalink, url.provider, url.url)
            if value in result:
                raise RuntimeError(f'Value duplicates: {value}')
            result.add(value)
        return result

    def get_input_permalink_ids(self, input_permalinks_table: str) -> List[uint]:
        result = set()
        for row in self.persistence_manager.read(input_permalinks_table):
            value = row['id']
            if value in result:
                raise RuntimeError(f'Value duplicates: {value}')
            result.add(value)
        return list(result)

    @staticmethod
    def get_new_mapping(permalink_id: uint, operators: List[str]) -> StorageMapping:
        operator_id = random.choice(operators)
        orig_hotel_id = str(random.randint(0, ORIG_HOTEL_ID_MAX))
        room_name = f'room_{random.randint(0, PERMAROOM_INDEX_MAX)}'
        return StorageMapping(
            is_deleted=False,
            need_new_permaroom=False,
            permalink=permalink_id,
            operator_id=operator_id,
            orig_hotel_id=orig_hotel_id,
            mapping_key=room_name,
            permaroom_id=None,
            orig_room_name=room_name,
            count=random.randint(0, OFFER_COUNT_MAX),
            prices_per_night='prices_per_night_value',
            requests_range='requests_range_value',
            price=random.randint(0, PRICE_MAX),
            url=f'https://{operator_id}/{orig_hotel_id}',
            params='offer_params',
            comment='offer_comment',
            status_yang_mappings=StageStatus.NOTHING_TO_DO,
        )

    def get_permalink_mappings(
        self,
        permalink_id: uint,
        known_mappings: Set[tuple],
        operators: List[str],
    ) -> Generator[Dict[str, Any]]:
        count = 0
        mappings_count = random.randrange(NEW_MAPPINGS_MIN, NEW_MAPPINGS_MAX)
        while count < mappings_count:
            mapping = self.get_new_mapping(permalink_id, operators)
            key = MappingKey.get_key(mapping)
            if key in known_mappings:
                continue
            yield dc_to_dict(mapping)
            count += 1

    def get_new_mappings(
        self,
        permalink_ids: List[uint],
        known_mappings: Set[tuple],
        operators: List[str],
    ) -> List[Dict[str, Any]]:
        new_mappings = list()
        for permalink in permalink_ids:
            new_mappings.extend(self.get_permalink_mappings(permalink, known_mappings, operators))
        return new_mappings

    @staticmethod
    def get_permalink_urls(
        permalink_id: uint,
        known_urls: Set[tuple],
        providers: List[str],
    ) -> Generator[Dict[str, Any]]:
        urls_count = random.randrange(len(providers))
        for _ in range(urls_count):
            provider = random.choice(providers)
            url = StorageUrl(
                permalink=permalink_id,
                provider=provider,
                url=f'https://{provider}/{permalink_id}/{random.randint(0, ORIG_HOTEL_ID_MAX)}'
            )
            key = UrlKey.get_key(url)
            if key in known_urls:
                continue
            yield dc_to_dict(url)

    def get_new_urls(
        self,
        permalink_ids: List[uint],
        known_urls: Set[tuple],
        providers: List[str],
    ) -> List[Dict[str, Any]]:
        new_urls = list()
        for permalink in permalink_ids:
            new_urls.extend(self.get_permalink_urls(permalink, known_urls, providers))
        return new_urls

    def run(self):
        LOG.info(f'Processing {self.input_path}')
        random.seed()

        operators = [o['OperatorId'] for o in self.persistence_manager.read('//home/travel/prod/config/operators')]
        providers = [p['Code'] for p in self.persistence_manager.read('//home/travel/prod/config/partners')]

        input_permalinks_table = self.persistence_manager.join(self.input_path, 'permalinks')

        output_permalinks_table = self.persistence_manager.join(self.output_path, 'permalinks')
        output_urls_table = self.persistence_manager.join(self.output_path, 'urls')
        output_mappings_table = self.persistence_manager.join(self.output_path, 'mappings')

        known_mappings = self.get_known_mappings()
        LOG.info(f'Got {len(known_mappings)} known mappings')

        known_urls = self.get_known_urls()
        LOG.info(f'Got {len(known_urls)} known urls')

        input_permalinks = self.get_input_permalink_ids(input_permalinks_table)
        LOG.info(f'Got {len(input_permalinks)} input permalinks')

        new_mappings = self.get_new_mappings(input_permalinks, known_mappings, operators)
        LOG.info(f'Got {len(new_mappings)} new mappings')

        new_urls = self.get_new_urls(input_permalinks, known_urls, providers)
        LOG.info(f'Got {len(new_urls)} new urls')

        LOG.info(f'Copying permalinks from {input_permalinks_table} to {output_permalinks_table}')
        self.persistence_manager.copy(input_permalinks_table, output_permalinks_table)

        LOG.info(f'Writing mappings to {output_mappings_table}')
        self.persistence_manager.write(output_mappings_table, new_mappings, get_dc_yt_schema(StorageMapping))

        LOG.info(f'Writing urls to {output_urls_table}')
        self.persistence_manager.write(output_urls_table, new_urls, get_dc_yt_schema(StorageUrl))

        LOG.info('All done')
