# -*- coding: utf-8 -*-

from datetime import date

from dateutil.relativedelta import relativedelta

from travel.hotels.content_manager.lib.processor import Processor


class ProcessorPartnerOffers(Processor):

    # noinspection SpellCheckingInspection
    def get_hotel_ids(self, table, hotel_ids_table):
        query = f'''
            $cluster_permalinks = (
                SELECT t1.cluster_permalink AS cluster_permalink, t2.id AS permalink
                FROM `//home/travel/prod/general/altay_mappings/latest/permalink_to_cluster_permalink` AS t1
                JOIN  `{table}` AS t2 ON (t1.permalink == t2.id)
            );

            $result_with_partner_id_int = (
                SELECT t2.permalink AS permalink, t1.partnerid AS PartnerIdInt, t1.originalid AS original_id
                FROM `//home/travel/prod/general/altay_mappings/latest/partnerid_originalid_to_cluster_permalink` AS t1
                JOIN $cluster_permalinks AS t2 ON (t1.permalink == t2.cluster_permalink)
            );

            $result_with_partner_id = (
                SELECT permalink, t2.PartnerId AS partner_id, original_id
                FROM $result_with_partner_id_int AS t1
                JOIN `//home/travel/prod/config/partners` AS t2 USING (PartnerIdInt)
            );

            INSERT INTO `{hotel_ids_table}` WITH TRUNCATE
            SELECT * FROM $result_with_partner_id;
        '''
        self.yql_client.run_query(query)

    @staticmethod
    def parse_date(dt_str):
        return date(int(dt_str[0:4]), int(dt_str[5:7]), int(dt_str[8:10]))

    def scan_logs(self, hotel_ids_table, table_out, date_from, date_to):
        query = f'''
            PRAGMA yson.DisableStrict;
            PRAGMA yson.AutoConvert;

            $extract_offers_data = ($x) -> {{
                $offers = Yson::ConvertToList(Yson::Lookup(Yson::Lookup($x, "Offers"), "Offer"));
                $om = ListMap($offers,
                ($y) -> {{ RETURN AsTuple(
                    Yson::LookupString($y, "OperatorId"),
                    Yson::LookupString(Yson::Lookup($y, "CatRoomData"), "MappingKey"),
                    Yson::LookupString($y, "DisplayedTitle"),
                    Yson::LookupInt64(Yson::Lookup($y, "Price"), "Amount"),
                    Yson::LookupString(Yson::Lookup($y, "LandingInfo"), "LandingPageUrl"),
                    Yson::LookupUint64(Yson::Lookup(Yson::Lookup($y, "CatRoomData"), "PermalinkData"), "Id"),
                    Yson::LookupString($y, "Id")); }});
                $of = ListFilter($om,
                ($y) -> {{
                    RETURN $y.0 IS NOT NULL
                    AND $y.1 IS NOT NULL
                    AND $y.2 IS NOT NULL
                    AND $y.3 IS NOT NULL
                    AND $y.4 IS NOT NULL
                    AND $y.5 IS NOT NULL
                    AND $y.6 IS NOT NULL; }});
                RETURN $of;
            }};
            $searcher_logs = (
                SELECT
                    PartnerId AS partner_id,
                    OriginalId AS orig_hotel_id,
                    CheckInDate as check_in,
                    CheckOutDate as check_out,
                    Currency as currency,
                    Occupancy as occupancy,
                    iso_eventtime as request_ts,
                    DateTime::ToDays(CAST(CheckOutDate AS Date) - CAST(CheckInDate AS Date))
                        as nights,
                    $extract_offers_data(Result) as res
                FROM RANGE(`logs/travel-hotels-search-result-log/1d`, `{date_from}`, `{date_to}`)
            );

            $flat_logs = (
                SELECT
                    partner_id AS partner_id,
                    orig_hotel_id AS orig_hotel_id,
                    check_in as check_in,
                    check_out as check_out,
                    currency as currency,
                    occupancy as occupancy,
                    request_ts as request_ts,
                    nights as nights,
                    res.0 as operator_id,
                    res.1 as mapping_key,
                    res.2 as orig_room_name,
                    res.3 / nights as price,
                    res.4 as url,
                    CAST(res.5 AS Uint64) as permalink,
                    res.6 as offer_id
                FROM $searcher_logs
                FLATTEN LIST BY res
            );

            $filtered_logs = (
                SELECT
                    orig_hotel_id,
                    check_in,
                    check_out,
                    currency,
                    occupancy,
                    request_ts,
                    nights,
                    operator_id,
                    mapping_key,
                    orig_room_name,
                    price,
                    url,
                    hotels.permalink AS permalink,
                    offer_id
                FROM `{hotel_ids_table}` AS hotels
                LEFT JOIN $flat_logs as logs
                ON
                    hotels.partner_id == logs.partner_id AND
                    hotels.original_id == logs.orig_hotel_id
            );

            $aggregated_logs = (
                SELECT
                    False AS is_deleted,
                    permalink,
                    operator_id,
                    orig_hotel_id,
                    mapping_key,
                    CAST(MIN(price / nights) AS String) || " - " || CAST(MAX(price / nights) AS String)
                        AS prices_per_night,
                    MIN(request_ts) || " - " || MAX(request_ts) AS requests_range,
                    MAX_BY(price, request_ts) as price,
                    MAX_BY(url, request_ts) as url,
                    NULL AS permaroom_id,
                    MAX_BY(orig_room_name, request_ts) AS orig_room_name,
                    COUNT(*) as count,
                    "CheckIn = " || MAX_BY(check_in, request_ts) ||
                        ", CheckOut = " || MAX_BY(check_out, request_ts) ||
                        ", Occupancy = " || MAX_BY(occupancy, request_ts) ||
                        ", Currency = " || MAX_BY(currency, request_ts)
                    AS params
                FROM $filtered_logs
                WHERE mapping_key IS NOT NULL
                GROUP BY (orig_hotel_id, permalink, operator_id, mapping_key)
            );

            INSERT INTO `{table_out}`
            SELECT * FROM $aggregated_logs
            ORDER BY permalink, operator_id, orig_hotel_id, mapping_key
        '''

        self.yql_client.run_query(query)

    def determine_urls(self, permalinks_table, permalink_urls_table):
        query = f'''
            PRAGMA yt.InferSchema;

            $permalinks = (
                SELECT id AS permalink FROM `{permalinks_table}`
            );

            $provider_ids = (
                SELECT id
                FROM `//home/altay/db/export/current-state/snapshot/provider`
                WHERE permalink LIKE 'ytravel%'
            );

            $provider_urls = (
                SELECT
                    permalink,
                    url,
                    provider
                FROM (
                    SELECT
                        clusters.permalink AS permalink,
                        ListExtract(data.company.urls, "value") AS urls,
                        providers.permalink AS provider
                    FROM `//home/altay/db/permalink-clusterization/signals-for-merge` AS clusters
                    JOIN `//home/altay/db/export/current-state/snapshot/provider` AS providers
                    ON (clusters.data.provider_id == providers.id)
                    WHERE clusters.permalink IN $permalinks AND clusters.data.provider_id IN $provider_ids
                ) FLATTEN LIST BY urls AS url
            );

            $partner_urls = (
                SELECT
                    permalink,
                    url,
                    String::Substring(provider, 8) AS provider
                FROM $provider_urls AS urls
                LEFT JOIN `//home/travel/prod/config/partners` AS partners
                ON (urls.provider == partners.Code)
                WHERE Url::Parse(url).Host == Url::Parse(Url).Host
            );

            $main_urls = (
                SELECT
                    permalink,
                    url
                FROM (
                    SELECT
                        permalink,
                        Yson::LookupString(url_info, "value") AS url,
                        Yson::LookupString(url_info, "type") AS provider
                    FROM (
                        SELECT
                            permalink,
                            Yson::ConvertToList(urls) AS urls
                        FROM hahn.`//home/altay/db/export/current-state/snapshot/company`
                        WHERE permalink IN $permalinks
                    ) FLATTEN LIST BY urls AS url_info
                )
                WHERE provider == "main"
            );

            $main_urls_unique = (
                SELECT
                    permalink,
                    SOME(url) AS url,
                    "main" AS provider
                FROM $main_urls
                GROUP BY permalink
            );

            $all_urls = (
                SELECT * FROM $partner_urls
                UNION ALL
                SELECT * FROM $main_urls_unique
            );

            INSERT INTO `{permalink_urls_table}` WITH TRUNCATE
            SELECT * FROM $all_urls
        '''
        self.yql_client.run_query(query)

    def run(self):
        # Input data
        input_permalinks_table = self.persistence_manager.join(self.input_path, 'permalinks')
        # Output data
        output_permalinks_table = self.persistence_manager.join(self.output_path, 'permalinks')
        output_urls_table = self.persistence_manager.join(self.output_path, 'urls')
        output_data_table = self.persistence_manager.join(self.output_path, 'mappings')
        # Temporary data
        temp_hotel_ids_table = self.persistence_manager.join(self.output_path, 'hotel_ids_table')
        # Storage data

        self.determine_urls(input_permalinks_table, output_urls_table)

        self.get_hotel_ids(input_permalinks_table, temp_hotel_ids_table)

        self.scan_logs(
            hotel_ids_table=temp_hotel_ids_table,
            table_out=output_data_table,
            date_from=str(date.today() - relativedelta(months=1)),
            date_to=str(date.today()),
        )

        self.persistence_manager.copy(input_permalinks_table, output_permalinks_table)

        if self.persistence_manager.exists(temp_hotel_ids_table):
            self.persistence_manager.delete(temp_hotel_ids_table)
