import logging

from travel.hotels.content_manager.lib.processor import Processor

LOG = logging.getLogger(__name__)


class ProcessorWLGetHotels(Processor):

    def get_hotels(self) -> None:
        input_table = self.persistence_manager.join(self.input_path, 'permalinks')
        output_permalinks_table = self.persistence_manager.join(self.output_path, 'permalinks')
        output_hotels_table = self.persistence_manager.join(self.output_path, 'hotels')

        query = f'''
            $cluster_permalinks = (
                SELECT DISTINCT
                    permalinks.permalink AS original_permalink,
                    clusters.cluster_permalink AS cluster_permalink,
                    permalinks.grouping_key AS grouping_key,
                FROM `{input_table}` AS permalinks
                LEFT JOIN `home/travel/prod/general/altay_mappings/latest/permalink_to_cluster_permalink` AS clusters
                ON permalinks.permalink == clusters.permalink
            );

            $all_permalinks = (
                SELECT DISTINCT
                    clusters.original_permalink AS original_permalink,
                    duplicates.permalink AS permalink,
                    clusters.grouping_key AS grouping_key,
                FROM $cluster_permalinks AS clusters
                LEFT JOIN `home/travel/prod/general/altay_mappings/latest/permalink_to_cluster_permalink` AS duplicates
                ON clusters.cluster_permalink == duplicates.cluster_permalink
            );

            $hotel_ids = (
                SELECT
                    permalinks.original_permalink AS permalink,
                    signals.provider_id AS provider_id,
                    signals.original_id AS original_id,
                    permalinks.grouping_key AS grouping_key,
                FROM $all_permalinks AS permalinks
                LEFT JOIN `home/altay/db/clusterization/state-current/signal_to_permalink` AS signals
                USING(permalink)
                WHERE original_id IS NOT NULL
            );

            $hotel_providers = (
                SELECT
                    hotels.permalink AS permalink,
                    providers.permalink AS provider_id,
                    hotels.original_id AS original_id,
                    hotels.grouping_key AS grouping_key,
                FROM $hotel_ids AS hotels
                LEFT JOIN `//home/altay/db/export/current-state/snapshot/provider` AS providers
                ON hotels.provider_id == providers.id
                WHERE providers.permalink IN (
                    "ytravel_bnovo",
                    "ytravel_dolphin",
                    "ytravel_expedia",
                    "ytravel_travelline",
                    "ytravel_bronevik",
                )
            );

            $hotels = (
                SELECT
                    hotels.permalink AS permalink,
                    map.PartnerId AS partner_id,
                    hotels.original_id AS original_id,
                    hotels.grouping_key AS grouping_key,
                FROM $hotel_providers AS hotels
                LEFT JOIN `home/travel/prod/config/partners` AS map
                ON hotels.provider_id == map.Code
            );

            INSERT INTO `{output_hotels_table}` WITH TRUNCATE
            SELECT *
            FROM $hotels;
        '''

        self.yql_client.run_query(query)
        LOG.info(f'Copying permalinks from {input_table} to {output_permalinks_table}')
        self.persistence_manager.copy(input_table, output_permalinks_table)

    def get_failed_permalinks(self) -> None:
        input_table = self.persistence_manager.join(self.input_path, 'failed_permalinks')
        output_permalinks_table = self.persistence_manager.join(self.output_path, 'failed_permalinks')
        query = f'''
            $cluster_permalinks = (
                SELECT DISTINCT
                    clusters.cluster_permalink AS cluster_permalink,
                    permalinks.grouping_key AS grouping_key,
                FROM `{input_table}` AS permalinks
                LEFT JOIN `home/travel/prod/general/altay_mappings/latest/permalink_to_cluster_permalink` AS clusters
                ON permalinks.permalink == clusters.permalink
            );

            INSERT INTO `{output_permalinks_table}` WITH TRUNCATE
            SELECT DISTINCT
                duplicates.permalink AS permalink,
                clusters.grouping_key AS grouping_key,
            FROM $cluster_permalinks AS clusters
            LEFT JOIN `home/travel/prod/general/altay_mappings/latest/permalink_to_cluster_permalink` AS duplicates
            ON clusters.cluster_permalink == duplicates.cluster_permalink
        '''

        self.yql_client.run_query(query)

    def run(self) -> None:
        self.get_hotels()

        input_failed_permalinks_table = self.persistence_manager.join(self.input_path, 'failed_permalinks')
        if self.persistence_manager.exists(input_failed_permalinks_table):
            self.get_failed_permalinks()
