# -*- coding: utf-8 -*-

from argparse import ArgumentParser
from datetime import datetime
import logging
import random

from yt.wrapper.client import YtClient

from travel.library.python.yandex_vault import resolve_secrets_ns
from travel.hotels.content_manager.lib.common import get_dc_yt_schema, ts_to_str_msk_tz
from travel.hotels.content_manager.lib.persistence_manager import YtPersistenceManager
from travel.hotels.content_manager.data_model.storage import TABLE_NAMES


def rand_bool(probability: float):
    return random.random() < probability


def main():
    logging.basicConfig(level=logging.INFO)
    parser = ArgumentParser()

    parser.add_argument('--vault-token', default=None)

    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)

    parser.add_argument('--path', required=True)
    parser.add_argument('--version', default=0)

    options = parser.parse_args()
    resolve_secrets_ns(options)

    version = options.version
    logging.info(f'{version=}')
    yt_client = YtClient(proxy=options.yt_proxy, token=options.yt_token)
    pm = YtPersistenceManager(yt_client, None)

    storage_snapshots_path = pm.join(options.path, 'storage_snapshots')
    logging.info(f'{storage_snapshots_path=}')
    storage_path = pm.join(options.path, 'storage')
    logging.info(f'{storage_path=}')
    start_ts = int(datetime.utcnow().timestamp())
    snapshot_name = ts_to_str_msk_tz(start_ts)
    logging.info(f'{snapshot_name=}')
    snapshot_path = pm.join(storage_snapshots_path, snapshot_name)
    logging.info(f'{snapshot_path=}')

    with pm.transaction():

        logging.info(f'Creating dir "{snapshot_path}"')
        pm.create_dir(snapshot_path)

        for key in TABLE_NAMES.keys():
            logging.info(f'Creating table "{key}"')
            pm.write(pm.join(snapshot_path, key), [], get_dc_yt_schema(TABLE_NAMES[key]))

        logging.info('Creating version table')
        pm.write(pm.join(snapshot_path, 'version'), [{'value': int(version)}])
        pm.link(storage_path, snapshot_path)

    logging.info('All done!')


if __name__ == '__main__':
    main()
