# -*- coding: utf-8 -*-

from argparse import ArgumentParser
from typing import Dict
import logging
import random

from yt.wrapper.client import YtClient

from travel.library.python.yandex_vault import resolve_secrets_ns
from travel.hotels.content_manager.data_model.stage import SCNewDescriptionData
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.persistence_manager import YtPersistenceManager


def get_sc_data(row: Dict[str, str]) -> SCNewDescriptionData:
    return SCNewDescriptionData(
        carrier_code=row['carrier_code'],
        car_type_code=row['car_type_code'],
        sc_code=row['sc_code'],
    )


def main():
    logging.basicConfig(level=logging.INFO)

    parser = ArgumentParser()

    parser.add_argument('--vault-token', default=None)

    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)

    parser.add_argument('--table-path', required=True)
    parser.add_argument('--count', type=int, default=100)

    options = parser.parse_args()
    resolve_secrets_ns(options)

    yt_client = YtClient(proxy=options.yt_proxy, token=options.yt_token)
    pm = YtPersistenceManager(yt_client, None)

    random.seed()

    dicts_path = '//home/travel/prod/train/service_classes'
    carriers_table_path = pm.join(dicts_path, 'carriers')
    car_types_table_path = pm.join(dicts_path, 'car_types')
    sc_table_path = pm.join(dicts_path, 'service_classes')

    carriers = set(row['carrier_code'] for row in pm.read(carriers_table_path) if row['enabled'])
    car_types = set(row['car_type_code'] for row in pm.read(car_types_table_path) if row['enabled'])
    sc_data_variants = (get_sc_data(row) for row in pm.read(sc_table_path))
    sc_data_variants = [sc for sc in sc_data_variants if sc.carrier_code in carriers and sc.car_type_code in car_types]
    assert options.count <= len(sc_data_variants)

    random.shuffle(sc_data_variants)
    data = (dc_to_dict(d) for d in sc_data_variants[:options.count])

    with pm.transaction():
        pm.write(options.table_path, data, get_dc_yt_schema(SCNewDescriptionData))


if __name__ == '__main__':
    main()
