# -*- coding: utf-8 -*-

from argparse import ArgumentParser
from typing import NamedTuple
import logging
import random

from yt.wrapper.client import YtClient

from travel.library.python.yandex_vault import resolve_secrets_ns
from travel.hotels.content_manager.data_model.stage import WLStartData
from travel.hotels.content_manager.data_model.types import (
    ActualizableAttribute, ActualizationStartReason, AssigneeSkill, ClusterizationStartReason, uint
)
from travel.hotels.content_manager.lib.common import dc_to_dict, get_dc_yt_schema
from travel.hotels.content_manager.lib.persistence_manager import YtPersistenceManager


PERMALINK_ID_MAX = 2 ** 64 - 1


class Route(NamedTuple):
    required_stages: str
    actualization_required_stages: str
    clusterization_required_stages: str


class Group(NamedTuple):
    route: Route
    actualization_start_reason: ActualizationStartReason
    clusterization_start_reason: ClusterizationStartReason
    assignee_skill: AssigneeSkill
    required_attributes: str
    grouping_key: str
    priority: uint


ROUTES = [
    Route(
        required_stages='actualization',
        actualization_required_stages='',
        clusterization_required_stages='',
    ),
    Route(
        required_stages='clusterization',
        actualization_required_stages='',
        clusterization_required_stages='',
    ),
    Route(
        required_stages='actualization',
        actualization_required_stages='clusterization',
        clusterization_required_stages='',
    ),
]


def rand_bool(probability: float):
    return random.random() < probability


def main():
    logging.basicConfig(level=logging.INFO)

    parser = ArgumentParser()

    parser.add_argument('--vault-token', default=None)

    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)

    parser.add_argument('--table-path', required=True)
    parser.add_argument('--count', type=int, default=100)

    options = parser.parse_args()
    resolve_secrets_ns(options)

    yt_client = YtClient(proxy=options.yt_proxy, token=options.yt_token)
    pm = YtPersistenceManager(yt_client, None)

    random.seed()

    group_count = int(options.count / 10) + 1
    groups = list()
    for n in range(group_count):
        attributes = list()
        for attribute in ActualizableAttribute:
            if rand_bool(.95):
                attributes.append(attribute.value)
        group = Group(
            route=random.choice(ROUTES),
            actualization_start_reason=random.choice(list(ActualizationStartReason)),
            clusterization_start_reason=random.choice(list(ClusterizationStartReason)),
            assignee_skill=random.choice(list(AssigneeSkill)),
            required_attributes=','.join(attributes),
            grouping_key=f'group_{n}',
            priority=uint(random.randint(0, group_count) + 1)
        )
        groups.append(group)

    data = list()
    for _ in range(options.count):
        permalink_id = random.randint(0, PERMALINK_ID_MAX)
        group = random.choice(groups)
        permalink = WLStartData(
            permalink=uint(permalink_id),
            required_stages=group.route.required_stages,
            actualization_required_stages=group.route.actualization_required_stages,
            clusterization_required_stages=group.route.clusterization_required_stages,
            hotel_name=f'hotel_name_{permalink_id}',
            requirements=f'requirements_{permalink_id}',
            required_attributes=group.required_attributes,
            actualization_start_reason=group.actualization_start_reason,
            clusterization_start_reason=group.clusterization_start_reason,
            assignee_skill=group.assignee_skill,
            grouping_key=group.grouping_key,
            priority=group.priority,
        )
        data.append(dc_to_dict(permalink))

    with pm.transaction():
        pm.write(options.table_path, data, get_dc_yt_schema(WLStartData))


if __name__ == '__main__':
    main()
