# -*- coding: utf-8 -*-
from __future__ import annotations

from argparse import ArgumentParser, Namespace
from pathlib import Path
import logging.config

import yaml

from travel.library.python.yandex_vault import resolve_secrets_ns
from travel.hotels.content_manager.lib.common import hide_secrets_ns
from travel.hotels.content_manager.lib.toloka_client import TolokaEnv, TolokaClient
from travel.hotels.content_manager.tools.update_toloka.log_config import LOG_CONFIG


LOG = logging.getLogger(__name__)


class TolokaUpdater(object):

    def __init__(self, options: Namespace) -> None:
        self.config_path: Path = options.config_path.expanduser()
        self.env: TolokaEnv = options.environment
        self.toloka_client = TolokaClient(options.environment, options.toloka_token)

    def dump_project(self, project_id: str):
        LOG.info('Getting project info')
        project = self.toloka_client.get_project(project_id)

        project_path = self.config_path / project_id
        LOG.info(f'Dumping project at {project_path}')
        project_path.mkdir(parents=True, exist_ok=True)

        with open(project_path / 'project.yaml', 'w') as f:
            yaml.dump(project.other_fields, f, allow_unicode=True)

        with open(project_path / 'template.css', 'w') as f:
            f.write(project.css)

        with open(project_path / 'template.html', 'w') as f:
            f.write(project.html)

        with open(project_path / 'template.js', 'w') as f:
            f.write(project.script)

        LOG.info('Dump finished')

    def update_project(self, project_path: Path) -> None:
        LOG.info(f'Updating project {project_path}')
        with open(project_path / 'environments.yaml') as f:
            environments_yaml = f.read()
        environments = yaml.safe_load(environments_yaml)

        with open(project_path / 'project.yaml') as f:
            project_yaml = f.read()
        project = yaml.safe_load(project_yaml)

        with open(project_path / 'template.css') as f:
            css = f.read()

        with open(project_path / 'template.html') as f:
            html = f.read()

        with open(project_path / 'template.js') as f:
            script = f.read()

        project_id = environments.get(self.env.value, {}).get('id')

        if project_id:
            self.toloka_client.update_project(project_id, css, html, script, project)

    def update(self) -> None:
        for project_path in self.config_path.glob('*'):
            self.update_project(project_path)


def main():
    logging.config.dictConfig(LOG_CONFIG)

    parser = ArgumentParser()
    parser.add_argument('--vault-token', default=None)

    parser.add_argument('--toloka-token', required=True)

    default_config_path = Path('~/projects/arcadia/travel/hotels/content_manager/yang_templates')
    parser.add_argument('--config-path', default=default_config_path, type=lambda p: Path(p))

    parser.add_argument('--environment', choices=[c for c in TolokaEnv], required=True, type=TolokaEnv)

    parser.add_argument('--dump-project')

    options = parser.parse_args()
    LOG.info(f'Unresolved options: {options}')
    resolve_secrets_ns(options)
    LOG.info(f'Resolved options: {hide_secrets_ns(options)}')

    updater = TolokaUpdater(options)

    if options.dump_project:
        updater.dump_project(options.dump_project)
        return

    updater.update()


if __name__ == '__main__':
    main()
