#include "worker.h"

#include "data.h"

#include <travel/hotels/devops/cfg_tool/config.pb.h>
#include <travel/hotels/lib/cpp/util/arcadia.h>
#include <travel/hotels/lib/cpp/data/data.h>
#include <travel/hotels/lib/cpp/protobuf/config.h>
#include <travel/hotels/lib/cpp/protobuf/tools.h>
#include <travel/hotels/lib/cpp/slack_forwarder_notificator/slack_forwarder_notificator.h>

#include <google/protobuf/util/message_differencer.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/colorizer/colors.h>

#include <util/generic/yexception.h>
#include <util/generic/hash_set.h>

namespace  {
    NColorizer::TColors& Colors = NColorizer::StdOut();
}


template <class TConfig>
TConfig TWorker::ReadProtoConfig(const TString& name) {
    TConfig pbCfg;
    bool read = false;
    TString ext = ".config";
    if (NTravel::OverrideConfigFromFile(&pbCfg, ConfigPath / (name + ext), "base")) {
        read = true;
    }
    if (NTravel::OverrideConfigFromFile(&pbCfg, ConfigPath / (name + "-" + Args.Env + ext), "env")) {
        read = true;
    }
    if (!pbCfg.IsInitialized()) {
        throw yexception() << "Message of type \"" << pbCfg.GetDescriptor()->full_name()
                           << "\" is missing required fields: " << pbCfg.InitializationErrorString();
    }
    if (!read) {
        throw yexception() << "No config files was found at path " << ConfigPath << Endl;
    }
    return pbCfg;
}

template <class TId, class TProto>
void TWorker::WriteConfig(const TMap<TId, TProto>& records, TYtCachePtr cache, const TString& ytTable) {
    INFO_LOG << Colors.Red() << cache->Proxy() << ": Writing table " << ytTable << "..." << Colors.OldColor() << Endl;
    NYT::TTableSchema schema = NTravel::NProtobuf::GenerateTableSchema(TProto(), true);
    TCachedTableDataPtr data = new TCachedTableData;
    for (const auto& [recId, record]: records) {
        NYT::TNode row;
        NTravel::NProtobuf::ProtoToNode(record, &row, true);
        data->Rows.push_back(row);
    }
    cache->Write(ytTable, schema, data);
    INFO_LOG << cache->Proxy() << ": Writing table " << ytTable << " - Done" << Endl;
}

template <class TId, class TProto>
bool TWorker::ReadConfig(TYtCachePtr cache, const TString& ytTable,  TMap<TId, TProto>* records) {
    INFO_LOG << cache->Proxy() << ": Reading table " << ytTable << "..." << Endl;
    auto data = cache->Read(ytTable);
    if (!data) {
        INFO_LOG << cache->Proxy() << ": Table " << ytTable << " does not exist" << Endl;
        return false;
    }
    for (const auto& row: data->Rows) {
        TProto record;
        NTravel::NProtobuf::NodeToProto(row, &record);
        TId id = NTravel::NProtobuf::GetIdentifier<TId, TProto>(record);
        (*records)[id] = record;
    }
    INFO_LOG << cache->Proxy() << ": Reading table " << ytTable << " - Done" << Endl;
    return true;
}

template <class TId, class TProto>
bool TWorker::CompareConfig(const TMap<TId, TProto>& localRecords, TYtCachePtr cache, const TString& ytTable) {
    TMap<TId, TProto> remoteRecords;
    if (!ReadConfig(cache, ytTable, &remoteRecords)) {
        return false;
    }
    bool equal = true;
    for (auto itLoc: localRecords) {
        auto itRem = remoteRecords.find(itLoc.first);
        if (itRem != remoteRecords.end()) {
            ::google::protobuf::util::MessageDifferencer differ;
            TString diff;
            differ.ReportDifferencesToString(&diff);
            if (!differ.Compare(itRem->second, itLoc.second)) {
                equal = false;
                Cout << Colors.Blue() << "* Record with id " << itLoc.first << " differs: " << Endl << diff << "******" << Colors.OldColor() << Endl;
            }
        } else {
            equal = false;
            Cout << Colors.Green() << "+++ Record with id " << itLoc.first << " is added" << Colors.OldColor() << Endl;
        }
    }
    for (auto itRem: remoteRecords) {
        auto itLoc = localRecords.find(itRem.first);
        if (itLoc == localRecords.end()) {
            equal = false;
            Cout << Colors.Red() << "--- Record with id " << itRem.first << " is deleted" << Colors.OldColor() << Endl;
        }
    }
    return equal;
}

template <class TId, class TProto>
void TWorker::ProcessConfig(const TMap<TId, TProto>& localRecords, TYtCachePtr cache, const TString& tableName) {
    TString ytTable = cache->Path() + "/" + tableName;
    bool equal = CompareConfig(localRecords, cache, ytTable);
    if (Action != EA_Push) {
        if (equal) {
            INFO_LOG << Colors.DarkWhite() << "No difference" << Colors.OldColor() << Endl;
        }
        return;
    }
    if (equal && !ForceWrite) {
        INFO_LOG << Colors.DarkWhite() << "No difference -> Skip writing" << Colors.OldColor() << Endl;
    } else {
        WriteConfig(localRecords, cache, ytTable);
    }
}

template <class TId, class TProto>
void TWorker::SetExplicitDefaultValues(TMap<TId, TProto>* data) {
    for (auto it = data->begin(); it != data->end(); ++it) {
        NTravel::NProtobuf::SetExplicitDefaultValues(&it->second);
    }
}

const TArguments& TArguments::SetupDefaultsAndCheck() {
    if (Env != "prod" && Env != "testing" && Env != "fake-env-for-tests") {
        throw yexception() << "Invalid env '" << Env << "'";
    }
    if (!YtProxies) {
        YtProxies.push_back("hahn");
        YtProxies.push_back("arnold");
    }
    if (!YtPath) {
        if (Env == "prod") {
            YtPath = "//home/travel/prod/config";
        } else if (Env == "testing") {
            YtPath = "//home/travel/testing/config";
        } else {
            throw yexception() << "yt-path not specified, and there is no default for environment '" << Env << "'";
        }
    }
    if (!ConfigDir) {
        ConfigDir = NTravel::GetArcadiaPath({"travel", "hotels", "devops", "config", "cfg_tool"});
    }
    // Some sanity check
    if (YtPath == "//home/travel/prod/config" && Env != "prod") {
        throw yexception() << "Cannot write non-prod env to prod path";
    }
    if (YtPath == "//home/travel/testing/config" && Env != "testing" && Env != "fake-env-for-tests") {
        throw yexception() << "Cannot write non-testing env to testing path";
    }
    return *this;
}

TWorker::TWorker(TArguments args)
    : Args(args.SetupDefaultsAndCheck())
    , ConfigPath(Args.ConfigDir)
{
}

void TWorker::Push(bool force) {
    Action = EA_Push;
    ForceWrite = force;
    DoWork();
}

void TWorker::Diff() {
    Action = EA_Diff;
    DoWork();
}

void TWorker::Validate() {
    Action = EA_Validate;
    DoWork();
}


void TWorker::PushInternal(NTravel::TSlackForwarderNotificator& notificator, const TString& notifyEnv, ui64 commitRevision) {
    ForceWrite = false;
    Action = EA_Push;
    CommitRevision = commitRevision;
    try {
        DoWork();
        notificator.ReportStatusNoThrow("DeployedTo" + notifyEnv);
    } catch (...) {
        ERROR_LOG << "Failed to deploy to " << notifyEnv << ", cause: " << CurrentExceptionMessage() << Endl;
        notificator.ReportStatusNoThrow("DeployTo" + notifyEnv + "Failed");
    }
}

void TWorker::DoWork() {
    INFO_LOG << "Working at " << Colors.ForeGreen() << Args.Env  << Colors.OldColor() << Endl;


    THashMap<TString/*Proxy*/, TYtCachePtr> caches;
    if (Action != EA_Validate) {
        // Prefetch yt
        for (const TString& ytProxy: Args.YtProxies) {
            auto cache = new TYtCache(ytProxy, Args.YtTokenPath, Args.YtToken, Args.YtPath);
            cache->Start();
            caches[ytProxy] = cache;
        }
    }

    NTravelProto::NConfig::TPartnersAndOperators pbPartnersAndOperatorsCfg = ReadProtoConfig<NTravelProto::NConfig::TPartnersAndOperators>("partners_and_operators");

    // Operators
    auto operators = NTravel::NProtobuf::MergeOverrides<NTravelProto::EOperatorId, NTravelProto::NConfig::TOperator>(pbPartnersAndOperatorsCfg.GetOperator());
    for (auto& [opId, op]: operators) {
        if (!op.HasPartnerId()) {
            throw yexception() << "Operator " << op.GetOperatorId()  << " has no associated partner";
        }
        if (op.GetGreenUrl().size() > 18) { // Вёрстка сказала, что больше 18 не входит
            throw yexception() << "Operator " << op.GetOperatorId()  << " has too long green url";
        }
        if (Args.EnableAllOperators) {
            op.SetEnabled(true);
        }
    }
    if (!Args.SkipAllOperatorsDefinedCheck) {
        for (int i = NTravelProto::EOperatorId_MIN; i <= NTravelProto::EOperatorId_MAX; ++i) {
            if (NTravelProto::EOperatorId_IsValid(i) && i != NTravelProto::OI_UNUSED && !operators.contains((NTravelProto::EOperatorId) i)) {
                throw yexception() << "Operator " << NTravelProto::EOperatorId(i) << " not listed in Config.Operators";
            }
        }
    }

    // Partners
    auto partners = NTravel::NProtobuf::MergeOverrides<NTravelProto::EPartnerId, NTravelProto::NConfig::TPartner>(pbPartnersAndOperatorsCfg.GetPartner());
    SetExplicitDefaultValues(&partners);
    if (!Args.SkipAllPartnersDefinedCheck) {
        for (int i = NTravelProto::EPartnerId_MIN; i <= NTravelProto::EPartnerId_MAX; ++i) {
            if (NTravelProto::EPartnerId_IsValid(i) && i != NTravelProto::PI_UNUSED && !partners.contains((NTravelProto::EPartnerId) i)) {
                throw yexception() << "Partner " << NTravelProto::EPartnerId(i) << " not listed in Config.Partners";
            }
        }
    }

    // OfferCache Clients
    NTravelProto::NConfig::TOfferCacheClients pbOfferCacheClientsCfg =
            ReadProtoConfig<NTravelProto::NConfig::TOfferCacheClients>("offercache_clients");
    auto ocClients = NTravel::NProtobuf::MergeOverrides<NTravel::TOfferCacheClientKey, NTravelProto::NConfig::TOfferCacheClient>(pbOfferCacheClientsCfg.GetClient());
    SetExplicitDefaultValues(&ocClients);

    // Hotels lists
    NTravelProto::NConfig::THotelLists pbHotelLists  =
            ReadProtoConfig<NTravelProto::NConfig::THotelLists>("hotel_lists");
    auto blackHotels = NTravel::NProtobuf::ReplaceOverrides<THotelEntryKey, NTravelProto::NConfig::THotelLists::THotelEntry>(pbHotelLists.GetBlacklistedHotel());
    auto whiteHotels = NTravel::NProtobuf::ReplaceOverrides<THotelEntryKey, NTravelProto::NConfig::THotelLists::THotelEntry>(pbHotelLists.GetWhitelistedHotel());
    auto wizardBanHotels = NTravel::NProtobuf::ReplaceOverrides<NTravel::TPermalink, NTravelProto::NConfig::THotelWizardBan>(pbHotelLists.GetWizardBanHotel());
    SetExplicitDefaultValues(&blackHotels);
    SetExplicitDefaultValues(&whiteHotels);

    // Payment schedule w/b lists
    NTravelProto::NConfig::TPaymentScheduleHotelList pbPaymentScheduleHotelList =
            ReadProtoConfig<NTravelProto::NConfig::TPaymentScheduleHotelList>("payment_schedule_hotel_list");
    auto paymentScheduleHotelList =  NTravel::NProtobuf::ReplaceOverrides<THotelEntryKey, NTravelProto::NConfig::TPaymentScheduleHotel>(pbPaymentScheduleHotelList.GetHotel());
    SetExplicitDefaultValues(&paymentScheduleHotelList);

    // Dolphin Tours, Rooms and Room Categories
    NTravelProto::NConfig::TDolphinPansions pbDolphinPansions =
            ReadProtoConfig<NTravelProto::NConfig::TDolphinPansions>("dolphin_pansions");
    auto dolphinPansions = NTravel::NProtobuf::ReplaceOverrides<ui64, NTravelProto::NConfig::TDolphinListRecord>(pbDolphinPansions.GetPansion());
    SetExplicitDefaultValues(&dolphinPansions);

    NTravelProto::NConfig::TDolphinRoomCats pbDolphinRoomCats =
            ReadProtoConfig<NTravelProto::NConfig::TDolphinRoomCats>("dolphin_room_cats");
    auto dolphinRoomCats = NTravel::NProtobuf::ReplaceOverrides<ui64, NTravelProto::NConfig::TDolphinListRecord>(pbDolphinRoomCats.GetRoomCat());
    SetExplicitDefaultValues(&dolphinRoomCats);

    NTravelProto::NConfig::TDolphinRooms pbDolphinRooms =
            ReadProtoConfig<NTravelProto::NConfig::TDolphinRooms>("dolphin_rooms");
    auto dolphinRooms = NTravel::NProtobuf::ReplaceOverrides<ui64, NTravelProto::NConfig::TDolphinListRecord>(pbDolphinRooms.GetRoom());
    SetExplicitDefaultValues(&dolphinRooms);

    NTravelProto::NConfig::TDolphinTours pbDolphinTours =
            ReadProtoConfig<NTravelProto::NConfig::TDolphinTours>("dolphin_tours");
    auto dolphinTours = NTravel::NProtobuf::ReplaceOverrides<ui64, NTravelProto::NConfig::TDolphinListRecord>(pbDolphinTours.GetTour());
    SetExplicitDefaultValues(&dolphinTours);

    //Dolphin room name normalize rules
    NTravelProto::NConfig::TNormalizeRules pbNormalizeRules =
            ReadProtoConfig<NTravelProto::NConfig::TNormalizeRules>("room_normalize_rules");
    auto normalizeRules = NTravel::NProtobuf::ReplaceOverrides<ui64, NTravelProto::NPartnerParsers::TNormalizeRule>(pbNormalizeRules.GetRule());
    SetExplicitDefaultValues(&normalizeRules);

    // SearchKey Restrictions
    NTravelProto::NConfig::TSearchKeyRestrictionsList pbSearchKeyRestrictionsList = ReadProtoConfig<NTravelProto::NConfig::TSearchKeyRestrictionsList>("searchkey_restrictions");
    auto searchKeyRestrictionsList = NTravel::NProtobuf::ReplaceOverrides<NTravel::TSearchKeyRestrictionsKey, NTravelProto::NConfig::TSearchKeyRestrictions>(pbSearchKeyRestrictionsList.GetEntry());
    // do not set default values , nulls are ok here

    // Region images list
    NTravelProto::NConfig::TRegionImageList pbRegionImageList  =
            ReadProtoConfig<NTravelProto::NConfig::TRegionImageList>("region_images");
    auto regionImages = NTravel::NProtobuf::ReplaceOverrides<TString, NTravelProto::NConfig::TRegionImage>(pbRegionImageList.GetImage());

    // Popular destinations
    NTravelProto::NConfig::TPopularDestinationsList pbPopularDestinationList  =
            ReadProtoConfig<NTravelProto::NConfig::TPopularDestinationsList>("popular_destinations");
    auto popularDestinations = NTravel::NProtobuf::ReplaceOverrides<TString, NTravelProto::NConfig::TPopularDestination>(pbPopularDestinationList.GetDest());

    // Mir hotels
    NTravelProto::NConfig::TMirHotels pbMirHotels  =
            ReadProtoConfig<NTravelProto::NConfig::TMirHotels>("mir_hotels");
    auto mirHotels = NTravel::NProtobuf::ReplaceOverrides<ui64, NTravelProto::NConfig::TMirHotel>(pbMirHotels.GetMirHotel());

    // Hotel images
    NTravelProto::NConfig::THotelImages pbHotelImages =
            ReadProtoConfig<NTravelProto::NConfig::THotelImages>("hotel_images");
    auto hotelImages = NTravel::NProtobuf::ReplaceOverrides<THotelImageKey, NTravelProto::NConfig::THotelImage>(pbHotelImages.GetImage());

    // Promo events
    NTravelProto::NConfig::TPromoEvents pbPromoEvents = ReadProtoConfig<NTravelProto::NConfig::TPromoEvents>("promo_events");
    auto promoEvents = NTravel::NProtobuf::ReplaceOverrides<TString, NTravelProto::NConfig::TPromoEvent>(pbPromoEvents.GetEvent());

    // Affiliate commission
    NTravelProto::NConfig::TAffiliatePartners pbAffiliatePartners = ReadProtoConfig<NTravelProto::NConfig::TAffiliatePartners>("affiliate_partners");
    auto affiliatePartners = NTravel::NProtobuf::ReplaceOverrides<TString, NTravelProto::NConfig::TAffiliatePartner>(pbAffiliatePartners.GetPartner());

    NTravelProto::NConfig::TAffiliatePartnerCommission pbAffiliatePartnerCommission =
            ReadProtoConfig<NTravelProto::NConfig::TAffiliatePartnerCommission>("affiliate_partner_commission");
    auto affiliatePartnerCommission = NTravel::NProtobuf::ReplaceOverrides<TAffiliatePartnerCommissionKey, NTravelProto::NConfig::TAffiliatePartnerCommissionItem>(pbAffiliatePartnerCommission.GetCommissionItem());

    NTravelProto::NConfig::TAffiliateUserCommission pbAffiliateUserCommission = ReadProtoConfig<NTravelProto::NConfig::TAffiliateUserCommission>("affiliate_user_commission");
    auto affiliateUserCommission = NTravel::NProtobuf::ReplaceOverrides<TAffiliateUserCommissionKey, NTravelProto::NConfig::TAffiliateUserCommissionItem>(pbAffiliateUserCommission.GetCommissionItem());

    // Black Friday 2021 hotels
    NTravelProto::NConfig::TBlackFriday2021HotelList pbBf2021Hotels  =
            ReadProtoConfig<NTravelProto::NConfig::TBlackFriday2021HotelList>("black_friday_2021_hotel_list");
    auto bf2021Hotels = NTravel::NProtobuf::ReplaceOverrides<NTravel::THotelId, NTravelProto::NConfig::TBlackFriday2021Hotel>(pbBf2021Hotels.GetHotel());

    NTravelProto::NConfig::TGoogleHotelsBlackList pbGoogleHotelsBlackList =
            ReadProtoConfig<NTravelProto::NConfig::TGoogleHotelsBlackList>("google_hotels_black_list");
    auto googleHotelsBlackList = NTravel::NProtobuf::ReplaceOverrides<TString, NTravelProto::NConfig::TGoogleHotelsBlackListItem>(pbGoogleHotelsBlackList.GetBlackListItem());

    if (Action == EA_Validate) {
        return;
    }

    // Do other actions
    bool anyOk = false;
    for (const auto& [ytProxy, cache]: caches) {
        if (CommitRevision) {
            auto actualRevision = cache->GetRevision();
            if (CommitRevision.GetRef() < actualRevision) {
                ERROR_LOG << Colors.Blue() << "Skip action at YtProxy " << ytProxy << ", because revision in YT " << actualRevision <<
                             " is > than current " << CommitRevision <<  Colors.OldColor() << Endl;
                continue;
            }
        }
        try {
            ProcessConfig(operators, cache, "operators");
            ProcessConfig(partners, cache, "partners");
            ProcessConfig(ocClients, cache, "offercache_clients");
            ProcessConfig(whiteHotels, cache, "hotels_whitelist");
            ProcessConfig(blackHotels, cache, "hotels_blacklist");
            ProcessConfig(wizardBanHotels, cache, "hotels_wizard_ban");
            ProcessConfig(dolphinPansions, cache, "dolphin_pansions");
            ProcessConfig(dolphinRoomCats, cache, "dolphin_room_cats");
            ProcessConfig(dolphinRooms, cache, "dolphin_rooms");
            ProcessConfig(dolphinTours, cache, "dolphin_tours");
            ProcessConfig(normalizeRules, cache, "room_normalize_rules");
            ProcessConfig(searchKeyRestrictionsList, cache, "searchkey_restrictions");
            ProcessConfig(regionImages, cache, "region_images");
            ProcessConfig(popularDestinations, cache, "popular_destinations");
            ProcessConfig(paymentScheduleHotelList, cache, "payment_schedule_hotel_list");
            ProcessConfig(mirHotels, cache, "mir_hotels");
            ProcessConfig(hotelImages, cache, "hotel_images");
            ProcessConfig(promoEvents, cache, "promo_events");
            ProcessConfig(affiliatePartners, cache, "affiliate_partners");
            ProcessConfig(affiliatePartnerCommission, cache, "affiliate_partner_commission");
            ProcessConfig(affiliateUserCommission, cache, "affiliate_user_commission");
            ProcessConfig(bf2021Hotels, cache, "black_friday_2021_hotel_list");
            ProcessConfig(googleHotelsBlackList, cache, "google_hotels_black_list");
            anyOk = true;
        } catch (...) {
            ERROR_LOG << Colors.Red() << "Failed to do action at YtProxy " << ytProxy << ", exception: " << CurrentExceptionMessage() << Colors.OldColor() << Endl;
        }
    };
    if (!anyOk) {
        throw yexception() << "Totally failed to do action";
    }

    // Commiting!
    anyOk = false;
    for (const auto& [ytProxy, cache]: caches) {
        if (CommitRevision) {
            cache->SetRevision(CommitRevision.GetRef());
        }
        if (cache->Commit()) {
            anyOk = true;
        }
    }
    if (!anyOk) {
        throw yexception() << "Totally failed to do action";
    }
}
