#include "yt_cache.h"

#include <mapreduce/yt/util/ypath_join.h>

#include <library/cpp/colorizer/colors.h>
#include <library/cpp/logger/global/global.h>

namespace {
    NColorizer::TColors& Colors = NColorizer::StdOut();
}

static const TString YT_REVISION_ATTRIBUTE = "@_arcadia_revision";

TYtCache::TYtCache(const TString& proxy, const TString& tokenPath, const TString& token, const TString& path)
    : Proxy_(proxy)
    , Path_(path)
    , Client_(NYT::CreateClient(proxy, NYT::TCreateClientOptions().TokenPath(tokenPath).Token(token)))
    , Tx_(Client_->StartTransaction())
    , ActiveWriters_(0)
{

}

TYtCache::~TYtCache() {
    Stop();
}

void TYtCache::Start() {
    Threads_.Start(16);
    TGuard g(Lock_);
    try {
        if (!Client_->Exists(Path_)) {
            INFO_LOG << Proxy_ << ".`" << Path_ << "` does not exist" << Endl;
            return;
        }
        const auto& nodes = Client_->List(Path_, NYT::TListOptions().MaxSize(1000).AttributeFilter(NYT::TAttributeFilter().AddAttribute("type")));
        for (const auto& node: nodes) {
            if (node.GetAttributes().At("type").AsString() != "table") {
                continue;
            }
            TString tableName = NYT::JoinYPaths(Path_, node.AsString());
            TReadTableInfoPtr tableInfo = new TReadTableInfo();
            ReadTables_[tableName] = tableInfo;
            Threads_.SafeAddFunc([this, tableName, tableInfo]() {
                try {
                    tableInfo->Data = new TCachedTableData;
                    for (auto reader = Client_->CreateTableReader<NYT::TNode>(tableName); reader->IsValid(); reader->Next()) {
                        tableInfo->Data->Rows.push_back(reader->GetRow());
                    }
                } catch (...) {
                    tableInfo->ReadError = CurrentExceptionMessage();
                }
                tableInfo->ReadFinished.Set();
            });
        }
    } catch (...) {
        CommonReadError_ = CurrentExceptionMessage();
    }
}

void TYtCache::Stop() {
    Threads_.Stop();
}

ui64 TYtCache::GetRevision() {
    try {
        NYT::TNode node = Tx_->Get(NYT::JoinYPaths(Path_, YT_REVISION_ATTRIBUTE));
        return node.IntCast<ui64>();
    } catch (...) {
        ERROR_LOG << "Failed to get revision at " << Proxy_ << ", cause is " << CurrentExceptionMessage() << Endl;
        return 0;
    }
}

void TYtCache::SetRevision(ui64 revision) {
    NYT::TNode node(revision);
    Tx_->Set(NYT::JoinYPaths(Path_, YT_REVISION_ATTRIBUTE), node);
}

TCachedTableDataPtr TYtCache::Read(const TString& tablePath) const {
    TReadTableInfoPtr info;
    with_lock (Lock_) {
        if (CommonReadError_) {
            throw yexception() << "Failed to list: " << CommonReadError_;
        }
        auto it = ReadTables_.find(tablePath);
        if (it == ReadTables_.end()) {
            return nullptr;
        }
        info = it->second;
    }
    while (!info->ReadFinished) {
        Sleep(TDuration::MilliSeconds(50));
    }
    if (info->ReadError) {
        throw yexception() << "Failed to read '" << tablePath << "': " << info->ReadError;
    }
    return info->Data;
}

void TYtCache::Write(const TString& tablePath, const NYT::TTableSchema& schema, TCachedTableDataPtr data) {
    with_lock (Lock_) {
        ++ActiveWriters_;
        TWriteTableInfoPtr tableInfo = new TWriteTableInfo;
        WriteTables_[tablePath] = tableInfo;
        Threads_.SafeAddFunc([this, tablePath, schema, data, tableInfo]() {
            try {
                if (Tx_->Exists(tablePath)) {
                    Tx_->Remove(tablePath);
                }
                auto node = NYT::TNode()("schema", schema.ToNode());
                Tx_->Create(tablePath, NYT::NT_TABLE, NYT::TCreateOptions().Recursive(true).Attributes(node));
                auto writer = Tx_->CreateTableWriter<NYT::TNode>(tablePath);
                for (const NYT::TNode& row: data->Rows) {
                    writer->AddRow(row);
                }
            } catch (...) {
                tableInfo->WriteError = CurrentExceptionMessage();
            }
            with_lock (Lock_) {
                --ActiveWriters_;
            }
        });
    }
}

bool TYtCache::Commit() {
    while (true) {
        with_lock (Lock_) {
            if (ActiveWriters_ == 0) {
                break;
            }
        }
        Sleep(TDuration::MilliSeconds(50));
    }
    try {
        Tx_->Commit();
    } catch (...) {
        ERROR_LOG << Colors.Red() << "Failed to commit at " << Proxy_ << ", exception: " << CurrentExceptionMessage() << Colors.OldColor() << Endl;
        return false;
    }

    bool ok = true;
    for (const auto& [tableName, tableInfo]: WriteTables_) {
        if (tableInfo->WriteError) {
            ok = false;
            ERROR_LOG << Colors.Red() << "Failed to write table " << Proxy_ << ".`" << tableName << "`, exception: " << tableInfo->WriteError << Colors.OldColor() << Endl;
        }
    }
    return ok;
}
