# -*- coding: utf-8 -*-

from __future__ import print_function

import argparse
import yt.wrapper as yt

from library.python.protobuf.yql import yql_proto_field
from travel.hotels.proto2.hotels_pb2 import TInventoryList, TVersionList
from travel.proto.commons_pb2 import TJson

TABLE_SCHEMAS = {
    'inventory':
        [
            {
                "name": "HotelId",
                "type": "string",
                "sort_order": "ascending",
                "required": True
            },
            {
                "name": "Version",
                "type": "uint64",
                "required": True
            },
            {
                "name": "Inventory",
                "type": "string",
                "required": True
            },
            {
                "name": "ActualizationTimestamp",
                "type": "uint64",
                "required": True
            }
        ],
    'offer_availability': [
        {
            "name": "HotelId",
            "type": "string",
            "sort_order": "ascending",
            "required": True
        },
        {
            "name": "CheckIn",
            "type": "uint64",
            "sort_order": "ascending",
            "required": True
        },
        {
            "name": "CheckOut",
            "type": "uint64",
            "sort_order": "ascending",
            "required": True

        },
        {
            "name": "Response",
            "type": "string",
            "required": True
        },
        {
            "name": "Versions",
            "type": "string",
            "required": True
        },
        {
            "name": "ActualizationTimestamp",
            "type": "uint64",
            "required": True
        }
    ]
}

EXTRA_ATTRIBUTES = {
    'inventory': [
        ('Inventory', TInventoryList)

    ],
    'offer_availability': [
        ('Response', TJson),
        ('Versions', TVersionList)
    ]
}

CLUSTER_NAME_TO_BUNDLE = {
    'hahn': 'travel-prod',
    'arnold': 'travel-prod',
    'seneca-sas': 'travel',
    'seneca-vla': 'travel',
    'seneca-man': 'travel'
}


def create_dir(cluster, path):
    client = yt.YtClient(cluster, config={"backend": "rpc"})
    if not client.exists(path):
        print('Creating dir', path, "at cluster", cluster)
        client.create('map_node', path)


def create_replica(meta_cluster, replica_cluster, table_path, replica_path, schema, set_bundles, attributes):
    meta_client = yt.YtClient(meta_cluster, config={"backend": "rpc"})
    replica_client = yt.YtClient(replica_cluster, config={"backend": "rpc"})
    print('Creating table replica to replicate from', meta_cluster, table_path, "to", replica_cluster, replica_path)
    replica_id = meta_client.create("table_replica", attributes={
        "table_path": table_path,
        "cluster_name": replica_cluster,
        "replica_path": replica_path,
    })
    print("Replica id is", replica_id)
    print("Creating replica table at", replica_cluster, replica_path)
    replica_client.create("table", replica_path, ignore_existing=True, attributes={
        "schema": schema,
        "dynamic": True,
        "upstream_replica_id": replica_id
    })
    if set_bundles:
        bundle = CLUSTER_NAME_TO_BUNDLE.get(replica_cluster)
        if bundle is not None:
            print('Setting bundle for', replica_path, 'at cluster', replica_cluster)
            replica_client.set(yt.ypath_join(replica_path, '@tablet_cell_bundle'), bundle)
    if attributes:
        for name, descriptor in attributes:
            replica_client.set_attribute(replica_path, '_yql_proto_field_' + name, yql_proto_field(descriptor))
    print('Mounting', replica_path, "at cluster", replica_cluster)
    replica_client.mount_table(replica_path, sync=True)
    print('Enabling replica', replica_id)
    meta_client.alter_table_replica(replica_id, True, mode="async")


def create_single_table(cluster, path, table_name, schema, args):
    client = yt.YtClient(cluster, config={"backend": "rpc"})
    table_path = yt.ypath_join(path, table_name)
    if client.exists(table_path):
        if args.force or raw_input('The table {} already exists. Do you want to recreate it?\n'
                                   'Type "YES" to confirm: '.format(table_path)).upper() == "YES":
            remove_table(cluster, path, table_name, True)
        else:
            return
    create_dir(cluster, path)
    print('Creating table', table_path, "at cluster", cluster)
    client.create('table', table_path, recursive=True, attributes={
        'dynamic': True,
        'schema': schema
    })
    if args.set_bundles:
        bundle = CLUSTER_NAME_TO_BUNDLE.get(cluster)
        if bundle is not None:
            print('Setting bundle for', table_path)
            client.set(yt.ypath_join(table_path, '@tablet_cell_bundle'), bundle)
    print('Mounting', table_path, "at cluster", cluster)
    client.mount_table(table_path, sync=True)

def create_replicated_table(meta_cluster, replica_clusters, path, table_name, schema, args):
    client = yt.YtClient(meta_cluster, config={"backend": "rpc"})
    table_path = yt.ypath_join(path, table_name)
    if client.exists(table_path):
        if args.force or raw_input('The table {} already exists. Do you want to recreate it?\n'
                                   'Type "YES" to confirm: '.format(table_path)).upper() == "YES":
            remove_table(meta_cluster, path, table_name, True)
        else:
            return
    create_dir(meta_cluster, path)
    print('Creating replicated table', table_path, "at cluster", meta_cluster)
    client.create('replicated_table', table_path, recursive=True, attributes={
        'dynamic': True,
        'schema': schema,
        "replicated_table_options": {"enable_replicated_table_tracker": True},

    })
    if args.set_bundles:
        bundle = CLUSTER_NAME_TO_BUNDLE.get(meta_cluster)
        if bundle is not None:
            print('Setting bundle for', table_path)
            client.set(yt.ypath_join(table_path, '@tablet_cell_bundle'), bundle)
    print('Mounting', table_path, "at cluster", meta_cluster)
    client.mount_table(table_path, sync=True)
    for replica_cluster in replica_clusters:
        create_dir(replica_cluster, path)
        create_replica(meta_cluster, replica_cluster, table_path, table_path + '_replica', schema, args.set_bundles,
                       EXTRA_ATTRIBUTES.get(table_name))


def remove_single_table(cluster, path, table_name, force):
    table_path = yt.ypath_join(path, table_name)
    if force or raw_input('Do you really want to remove {} from {}?'
                          '\nType "YES" to confirm: '.format(table_path, cluster)).upper() == "YES":
        client = yt.YtClient(cluster, config={"backend": "rpc"})
        print("Removing table from", cluster, "at", table_path)
        client.remove(table_path)

def remove_table(meta_cluster, path, table_name, force):
    table_path = yt.ypath_join(path, table_name)
    if force or raw_input('Do you really want to remove {} from all clusters?'
                          '\nType "YES" to confirm: '.format(table_path)).upper() == "YES":
        print('Reading replicas', table_path)
        meta_client = yt.YtClient(meta_cluster, config={"backend": "rpc"})
        replicas = meta_client.get_attribute(table_path, 'replicas')
        for replica in replicas.values():
            print("Removing table replica from", replica["cluster_name"], "at", replica["replica_path"])
            replica_client = yt.YtClient(replica["cluster_name"], config={"backend": "rpc"})
            replica_client.remove(replica["replica_path"])
        print("Removing meta-table from", meta_cluster, "at", table_path)
        meta_client.remove(table_path)


def main():
    parser = argparse.ArgumentParser(add_help=True,
                                     description='Script for creating/removing L2 caches on several clusters')
    parser.add_argument('action', choices=('create', 'remove'), help='either create or remove cache')
    parser.add_argument('--meta-cluster', required=False, default='markov', help='names of meta-cluster')
    parser.add_argument('--replica-clusters', nargs='+', required=False,
                        default=['seneca-sas', 'seneca-man', 'seneca-vla'], help='names of replica cluster')
    parser.add_argument('--path', required=True, help='path to cache directory (will be created if does not exist)')
    parser.add_argument('--tables', nargs='+', choices=["inventory", "offer_availability"],
                        default=["inventory", "offer_availability"], help='cache tables to work with')
    parser.add_argument('--set-bundles', action='store_true', default=False, help='set bundles if the flag is on')
    parser.add_argument('--force', action='store_true', default=False, help='Do not aks for deletes')
    parser.add_argument('--no-replication', action='store_true', default=False, help='Do not create replicas - single table on metacluster')
    args = parser.parse_args()

    for table in args.tables:
        schema = TABLE_SCHEMAS.get(table)
        if schema is None:
            raise Exception('Unknown table name: "{}"'.format(table))
        if args.action == 'create':
            if args.no_replication:
                create_single_table(args.meta_cluster, args.path, table, schema, args)
            else:
                create_replicated_table(args.meta_cluster, args.replica_clusters, args.path, table, schema, args)
        elif args.action == 'remove':
            if args.no_replication:
                remove_single_table(args.meta_cluster, args.path, table, args.force)
            else:
                remove_table(args.meta_cluster, args.path, table, args.force)

if __name__ == '__main__':
    main()
