#!/usr/bin/env python2

from functools import wraps
import argparse
import json
import os
import requests
import stat
import subprocess
import time


def with_retries(func=None, max_attempts=10, backoff_factor=1.0, backoff_max=60, exc_info=True):
    if not func:
        return lambda f: with_retries(f, max_attempts=max_attempts)

    @wraps(func)
    def wrapper(*args, **kwargs):
        attempts = max_attempts
        while True:
            try:
                return func(*args, **kwargs)
            except Exception as e:
                attempts -= 1
                if attempts == 0:
                    raise
                print('%r failed, %d attempts left' % (func, attempts))
                if exc_info:
                    print(e)
                sleep_time = backoff_factor * 2 ** (max_attempts - attempts - 1)
                sleep_time = min(sleep_time, backoff_max)
                time.sleep(sleep_time)
    return wrapper


@with_retries
def get_task_info(resource_type, released_at):
    params = {
        'limit': 1,
        'type': resource_type,
        'state': 'READY',
        'attrs': json.dumps({'released': released_at})
    }
    rsp = requests.get('https://sandbox.yandex-team.ru/api/v1.0/resource', params=params)
    rsp.raise_for_status()
    result = None
    try:
        result = rsp.json()['items'][0]
    except (IndexError, KeyError):
        pass
    return result


@with_retries
def download_resource_from_sandbox(url, binary_name):
    subprocess.check_call(['wget', '--no-verbose', '-O', binary_name, url])


def download_resource(args):
    tasks = list()
    task_info = get_task_info(args.resource_type, 'stable')
    if task_info is not None:
        tasks.append(task_info)
    if args.released_at == 'testing':
        task_info = get_task_info(args.resource_type, 'testing')
        if task_info is not None:
            tasks.append(task_info)
    if not tasks:
        raise Exception('No resource {} ready for {} found'.format(args.resource_type, args.released_at))
    task_info = max(tasks, key=lambda x: x['time']['created'])
    download_url = task_info['http']['proxy'] + '/' + args.binary_name
    download_resource_from_sandbox(download_url, args.binary_name)


def hide_secrets(text):
    parts = text.split(' ')
    parts = ['*' * 10 if part.startswith('AQAD-') else part for part in parts]
    return ' '.join(parts)


def main():
    parser = argparse.ArgumentParser(description='rest args will pass to binary')
    released_choices = ['testing', 'stable']
    parser.add_argument('--resource-type', required=True)
    parser.add_argument('--binary-name', required=True)
    parser.add_argument('--plain-args')
    parser.add_argument('--secret-args')
    parser.add_argument('--released-at', choices=released_choices, default='stable')
    args = parser.parse_args()

    download_resource(args)
    os.chmod(args.binary_name, os.stat(args.binary_name).st_mode | stat.S_IEXEC)

    # Explicitly avoiding any escaping
    comm = './' + args.binary_name
    if args.secret_args:
        comm += ' ' + args.secret_args
    if args.plain_args:
        comm += ' ' + args.plain_args
    try:
        subprocess.check_call(comm, shell=True)
    except subprocess.CalledProcessError as e:
        raise subprocess.CalledProcessError(e.returncode, hide_secrets(e.cmd))


if __name__ == '__main__':
    main()
