package main

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/hotels/devops/partner_data_moderator/tg-bot/internal/moderator"
	"a.yandex-team.ru/travel/hotels/lib/go/ytstorage"
	"a.yandex-team.ru/travel/hotels/proto2"
	"context"
	"fmt"
	tb "gopkg.in/tucnak/telebot.v2"
	"reflect"
	"sort"
	"strconv"
	"strings"
	"time"
)

const (
	showButtonSlug               = "show_rateplan_button"
	hideButtonSlug               = "hide_rateplan_button"
	moreButtonSlug               = "more_rateplan_button"
	customTicketButtonSlug       = "custom_ticket_button"
	cancellationTicketButtonSlug = "cancellation_ticket_button"
)

const (
	cancellationComment = "Правила отмены (флаги) для яндекса не совпадают с текстовыми правилами отмены от отеля"
)

type botCfg struct {
	TGToken           string
	AllowedLogins     []string
	BaseYTPath        string
	YTCluster         string
	YTToken           string
	Night             *nightConfig
	RatePlanListLimit int
	LandingLabel      string
	LandingPrefix     string
}

type user struct {
	Login         string
	UserID        int
	Subscribed    bool
	AwaitsForNext bool
}

type chat struct {
	Name          string
	ID            int64
	AlertsEnabled bool
	AlertIsActive bool
}

type moderBot struct {
	moderator      *moderator.Moderator
	tg             *tb.Bot
	allowedLogins  map[string]bool
	users          map[int]*user
	alertChats     map[int64]*chat
	logger         log.Logger
	pendingReplies map[int]string
	cfg            *botCfg
}

func (b *moderBot) startListening() {
	b.tg.Start()
}

func (b *moderBot) handleStart(message *tb.Message) {
	if b.checkAllowed(message.Sender, "start") {
		u, exists := b.users[message.Sender.ID]
		if exists {
			if u.Subscribed {
				return
			}
			u.Subscribed = true
		} else {
			s := &user{
				Login:      message.Sender.Username,
				UserID:     message.Sender.ID,
				Subscribed: true,
			}
			b.users[s.UserID] = s
		}
		_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID},
			"Буду присылать тарифные планы на модерацию. Отказаться от их получения можно командой /stop")
		b.saveUsers(context.Background())
		go b.next(message.Sender.ID, false)
	}
}

func (b *moderBot) handleStop(message *tb.Message) {
	if b.checkAllowed(message.Sender, "stop") {
		s, exists := b.users[message.Sender.ID]
		if exists {
			if !s.Subscribed {
				return
			}
			s.AwaitsForNext = false
			s.Subscribed = false
		}
		_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID},
			"Больше не буду писать про новые Тарифные Планы по мере их появления и при модерации существующих.\r\n"+
				"Вручную запросить ТП на проверку можно командой /check\r\n"+
				"Подписаться заново можно командой /start")
		b.saveUsers(context.Background())
	}
}

func (b *moderBot) handleCheck(message *tb.Message) {
	if b.checkAllowed(message.Sender, "check") {
		b.next(message.Sender.ID, false)
	}
}

func (b *moderBot) handleGet(message *tb.Message) {
	if b.checkAllowed(message.Sender, "get") {
		query := message.Payload
		parts := strings.Split(query, " ")
		if len(parts) != 2 || !strings.Contains(parts[0], ":") {
			_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID},
				"Указывайте параметры после get в формате `prefix:hotelID planID`")
			return
		}
		idParts := strings.Split(parts[0], ":")
		prefix := idParts[0]
		id := idParts[1]
		plan := parts[1]
		var partner proto2.EPartnerId
		switch prefix {
		case "tl":
			partner = proto2.EPartnerId_PI_TRAVELLINE
		case "bn":
			partner = proto2.EPartnerId_PI_BNOVO
		default:
			_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID},
				"Невалидный префикс: используйте 'tl' для travelline и 'bn' для bnovo")
			return
		}
		item := b.moderator.Get(moderator.HotelID{PartnerID: partner, OriginalID: id}, plan, true)
		if item == nil {
			_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID}, "Неизвестный тариф")
			return
		}
		b.sendRatePlan(message.Sender.ID, item, b.moderator.Stats().QueueSize)
	}
}

func (b *moderBot) handleList(message *tb.Message) {
	if b.checkAllowed(message.Sender, "list") {
		query := message.Payload
		parts := strings.Split(query, " ")
		if len(parts) != 1 || !strings.Contains(parts[0], ":") {
			_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID},
				"Указывайте параметры после list в формате `prefix:hotelID`")
			return
		}
		idParts := strings.Split(parts[0], ":")
		prefix := idParts[0]
		id := idParts[1]
		var partner proto2.EPartnerId
		switch prefix {
		case "tl":
			partner = proto2.EPartnerId_PI_TRAVELLINE
		case "bn":
			partner = proto2.EPartnerId_PI_BNOVO
		default:
			_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID},
				"Невалидный префикс: используйте 'tl' для travelline и 'bn' для bnovo")
			return
		}

		items, cropped := b.moderator.List(moderator.HotelID{PartnerID: partner, OriginalID: id}, b.cfg.RatePlanListLimit, true)
		if len(items) == 0 {
			_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID}, fmt.Sprintf("Тарифы не найдены (partnerId='%d', originalId='%s')", int(partner), id))
			return
		}
		for i := range items {
			b.sendRatePlan(message.Sender.ID, items[i], b.moderator.Stats().QueueSize)
		}
		if cropped {
			_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID}, fmt.Sprintf("Тарифов больше %d, показаны только первые %d", b.cfg.RatePlanListLimit, b.cfg.RatePlanListLimit))
		}
	}
}

func (b *moderBot) handleStats(message *tb.Message) {
	if b.checkAllowed(message.Sender, "stats") {
		s := b.moderator.Stats()

		type kv struct {
			Key   string
			Value int
		}

		var ss []kv
		for k, v := range s.Counter.ModeratedByUser {
			ss = append(ss, kv{k, v})
		}
		sort.Slice(ss, func(i, j int) bool {
			return ss[i].Value > ss[j].Value
		})

		msg := fmt.Sprintf(
			"Непроверенных тарифных планов: \r\n  - в очереди: %d \r\n  - всего: %d\r\n"+
				"Новых за сегодня: %d\r\n"+
				"Проверено за сегодня: %d\r\n", s.QueueSize, s.InProgressSize, s.Counter.NewRatePlans, s.Counter.ModeratedRatePlans)
		for _, kv := range ss {
			msg += fmt.Sprintf("Проверено @%s: %d\r\n", kv.Key, kv.Value)
		}
		_, _ = b.tg.Send(&tb.User{ID: message.Sender.ID}, msg)
	}
}

func (b *moderBot) handleStartAlerts(message *tb.Message) {
	if b.checkAllowed(message.Sender, "start alerts") {
		c, exists := b.alertChats[message.Chat.ID]
		if exists {
			if c.AlertsEnabled {
				return
			}
			c.AlertIsActive = false
			c.AlertsEnabled = true
		} else {
			var name string
			if message.Chat.Type == tb.ChatPrivate {
				name = "private chat with " + message.Chat.Username
			} else {
				name = message.Chat.Title
			}
			c = &chat{
				Name:          name,
				ID:            message.Chat.ID,
				AlertsEnabled: true,
			}
			b.alertChats[message.Chat.ID] = c
		}
		_, _ = b.tg.Send(message.Chat, "Включены уведомления о переполнении очереди модерации")
		b.saveChats(context.Background())
	}
}

func (b *moderBot) handleStopAlerts(message *tb.Message) {
	if b.checkAllowed(message.Sender, "stop alerts") {
		c, exists := b.alertChats[message.Chat.ID]
		if exists && c.AlertsEnabled {
			c.AlertsEnabled = false
			c.AlertIsActive = false
		} else {
			return
		}
		_, _ = b.tg.Send(message.Chat, "Уведомления о переполнении очереди модерации выключены")
		b.saveChats(context.Background())
	}
}

func (b *moderBot) handleAlert(size int, alert bool) {
	if b.cfg.Night != nil {
		s, err1 := time.Parse("15:04", b.cfg.Night.Start)
		e, err2 := time.Parse("15:04", b.cfg.Night.End)

		if err1 == nil && err2 == nil {
			now := time.Now()
			if (now.Hour() > s.Hour() && (s.Hour() > e.Hour() || now.Hour() < e.Hour())) ||
				(now.Hour() < e.Hour() && (s.Hour() > e.Hour() || now.Hour() > s.Hour())) ||
				(now.Hour() == s.Hour() && now.Minute() > s.Minute() && (s.Hour() != e.Hour() || now.Minute() < e.Minute())) ||
				(now.Hour() == e.Hour() && now.Minute() < s.Minute() && (s.Hour() != e.Hour() || now.Minute() > s.Minute())) {
				b.logger.Infof("Skipping alert since it's night time")
				return
			}
		}
	}
	for _, ch := range b.alertChats {
		if ch.AlertsEnabled {
			if alert {
				if !ch.AlertIsActive {
					ch.AlertIsActive = true
					_, _ = b.tg.Send(&tb.Chat{ID: ch.ID}, fmt.Sprintf("‼️ ALARM ‼️\r\n\r\nДлина очереди тарифных планов на модерацию превысила %d!", size))
				}
			} else {
				if ch.AlertIsActive {
					ch.AlertIsActive = false
					_, _ = b.tg.Send(&tb.Chat{ID: ch.ID}, fmt.Sprintf("✅ OK️\r\n\r\nДлина очереди тарифных планов на модерацию составляет %d", size))
				}
			}
		}
	}
	b.saveChats(context.Background())
}

func (b *moderBot) showButtonPressed(c *tb.Callback) {
	defer b.tg.Respond(c, &tb.CallbackResponse{})
	if b.checkAllowed(c.Sender, "show") {
		p, h, rp := parseData(c.Data)
		item := b.moderator.Get(moderator.HotelID{
			PartnerID:  p,
			OriginalID: h,
		}, rp, false)
		if item != nil {
			if item.State != moderator.Show {
				item.State = moderator.Show
				item.Moderator = c.Sender.Username
				item.VerifiedTimestamp = time.Now().Unix()
				b.moderator.Put(item)
				b.editRatePlan(c.Message, item, 0)
			}
		}
		if s, active := b.users[c.Sender.ID]; active && s.Subscribed {
			go b.next(c.Sender.ID, false)
		}
	}
}

func (b *moderBot) hideButtonPressed(c *tb.Callback) {
	defer b.tg.Respond(c, &tb.CallbackResponse{})
	if b.checkAllowed(c.Sender, "hide") {
		p, h, rp := parseData(c.Data)
		item := b.moderator.Get(moderator.HotelID{
			PartnerID:  p,
			OriginalID: h,
		}, rp, false)
		if item != nil {
			if item.State != moderator.Hide {
				item.State = moderator.Hide
				item.Moderator = c.Sender.Username
				item.VerifiedTimestamp = time.Now().Unix()
				b.moderator.Put(item)
				b.editRatePlan(c.Message, item, 2)
			}
		}
		if s, active := b.users[c.Sender.ID]; active && s.Subscribed {
			go b.next(c.Sender.ID, false)
		}
	}
}

func (b *moderBot) moreButtonPressed(c *tb.Callback) {
	defer b.tg.Respond(c, &tb.CallbackResponse{})
	if b.checkAllowed(c.Sender, "more") {
		p, h, rp := parseData(c.Data)
		item := b.moderator.Get(moderator.HotelID{
			PartnerID:  p,
			OriginalID: h,
		}, rp, false)
		if item != nil {
			b.moderator.Skip(item)
		}
		go b.next(c.Sender.ID, false)
	}
}

func (b *moderBot) cancellationTicketPressed(c *tb.Callback) {
	defer b.tg.Respond(c, &tb.CallbackResponse{})
	if b.checkAllowed(c.Sender, "cancellation-ticket") {
		p, h, rp := parseData(c.Data)
		item := b.moderator.Get(moderator.HotelID{
			PartnerID:  p,
			OriginalID: h,
		}, rp, false)
		if item != nil {
			item.Comment = cancellationComment
			item.Moderator = c.Sender.Username
			item.VerifiedTimestamp = time.Now().Unix()
			_, _ = b.tg.Send(c.Sender, fmt.Sprintf("Заведу тикет с текстом '%s', спасибо.\r\n"+
				"Не забудьте выбрать действие для этого тарифа", cancellationComment))
			b.moderator.Put(item)
		}
	}
}

func (b *moderBot) customTicketPressed(c *tb.Callback) {
	defer b.tg.Respond(c, &tb.CallbackResponse{})
	if b.checkAllowed(c.Sender, "custom-ticket") {
		p, h, rp := parseData(c.Data)
		item := b.moderator.Get(moderator.HotelID{
			PartnerID:  p,
			OriginalID: h,
		}, rp, false)
		if item != nil {
			msg, err := b.tg.Send(c.Sender, "Введите комментарий для заведения тикета",
				&tb.ReplyMarkup{
					ForceReply: true,
				})
			if err != nil {
				fmt.Println(err)
			} else {
				b.pendingReplies[msg.ID] = c.Data
			}
		}
	}
}

func (b *moderBot) replyReceived(message *tb.Message) {
	if b.checkAllowed(message.Sender, "reply") {
		if message.ReplyTo != nil {
			data, exists := b.pendingReplies[message.ReplyTo.ID]
			if exists {
				p, h, rp := parseData(data)
				item := b.moderator.Get(moderator.HotelID{
					PartnerID:  p,
					OriginalID: h,
				}, rp, false)
				if item != nil {
					item.Comment = message.Text
					item.Moderator = message.Sender.Username
					item.VerifiedTimestamp = time.Now().Unix()
					_, _ = b.tg.Send(message.Sender, fmt.Sprintf("Заведу тикет с текстом '%s', спасибо.\r\n"+
						"Не забудьте выбрать действие для этого тарифа", item.Comment))
					b.moderator.Put(item)
				}
			}
		}
	}
}

func (b *moderBot) handleTicketsCreated(user string, tickets []string) {
	for userID, u := range b.users {
		if u.Login == user {
			msg := "По вашим комментариям заведены тикеты:\r\n"
			for _, t := range tickets {
				msg += fmt.Sprintf(" - <a href='https://tracker.yandex.ru/%s'>%s</a>\r\n", t, t)
			}
			_, _ = b.tg.Send(&tb.User{ID: userID}, msg, &tb.SendOptions{ParseMode: tb.ModeHTML})
			return
		}
	}
}

func (b *moderBot) editRatePlan(msg *tb.Message, item *moderator.RatePlanItem, selectedButton int) {
	message := b.getItemDescription(item)
	rm := &tb.ReplyMarkup{}
	data := fmt.Sprintf("%d:%s:%s", item.PartnerID, item.OriginalID, item.RatePlanID)
	rm.InlineKeyboard = getButtons(data, selectedButton, item.PartnerID == proto2.EPartnerId_PI_BNOVO, 0)
	_, err := b.tg.Edit(msg, message, &tb.SendOptions{ReplyMarkup: rm, ParseMode: tb.ModeHTML, DisableWebPagePreview: true})
	if err != nil {
		b.logger.Warnf("unable to edit formatted message %s", message)
		_, err := b.tg.Edit(msg, message, &tb.SendOptions{ReplyMarkup: rm, ParseMode: tb.ModeDefault, DisableWebPagePreview: true})
		if err != nil {
			b.logger.Errorf("unable to edit plain message %s", message)
		}
	}
}

func (b *moderBot) sendRatePlan(userID int, item *moderator.RatePlanItem, count int) {
	user, found := b.users[userID]
	if !found {
		b.logger.Errorf("Unknown user with id %d", userID)
		return
	}
	b.logger.Infof("Will send a rate plan %s to user %s", item.ToString(), user.Login)
	u := &tb.User{ID: userID}
	message := b.getItemDescription(item)
	rm := &tb.ReplyMarkup{}
	data := fmt.Sprintf("%d:%s:%s", item.PartnerID, item.OriginalID, item.RatePlanID)
	index := -1
	switch item.State {
	case moderator.Show:
		index = 0
	case moderator.Warn:
		index = 0
	case moderator.Hide:
		index = 2
	}
	rm.InlineKeyboard = getButtons(data, index, item.PartnerID == proto2.EPartnerId_PI_BNOVO, count)
	_, err := b.tg.Send(u, message, &tb.SendOptions{ReplyMarkup: rm, ParseMode: tb.ModeHTML, DisableWebPagePreview: true})
	b.logger.Infof("Rate sent")
	if err != nil {
		b.logger.Warnf("unable to send formatted message %s", message)
		_, err = b.tg.Send(u, message, &tb.SendOptions{ReplyMarkup: rm, ParseMode: tb.ModeDefault, DisableWebPagePreview: true})
		if err != nil {
			b.logger.Errorf("unable to send plain message %s", message)
		}
	}
}

func (b *moderBot) checkAllowed(sender *tb.User, action string) bool {
	b.logger.Infof("%s runs %s", sender.Username, action)
	_, exists := b.allowedLogins[strings.ToLower(sender.Username)]
	if !exists {
		b.logger.Warnf("The user %s is not allowed in", sender.Username)
		_, _ = b.tg.Send(sender, "Я вас не знаю, извините")
		return false
	}
	if _, exists := b.users[sender.ID]; !exists {
		b.users[sender.ID] = &user{
			Login:         sender.Username,
			UserID:        sender.ID,
			Subscribed:    false,
			AwaitsForNext: false,
		}
	}
	return true
}

func (b *moderBot) next(userID int, wait bool) {
	user, found := b.users[userID]
	if !found {
		b.logger.Errorf("Unknown user with id %d", userID)
		return
	}
	b.logger.Infof("Next rate requested for user %s (with wait = %v)", user.Login, wait)
	rp, count, err := b.moderator.Next(wait)
	if err != nil {
		return
	}
	if rp == nil {
		b.logger.Infof("No new rateplans in the queue")
		if u, exists := b.users[userID]; exists && u.Subscribed {
			if !u.AwaitsForNext {
				b.logger.Infof("Setting user in AwaitsForNext state")
				_, _ = b.tg.Send(&tb.User{ID: userID}, "Нет новых тарифных планов. Я напишу, как они появятся")
				u.AwaitsForNext = true
				b.saveUsers(context.Background())
				go b.next(userID, true)
			} else {
				b.logger.Infof("User is already in AwaitsForNext state")
				_, _ = b.tg.Send(&tb.User{ID: userID}, "Все ещё нет новых тарифных планов. Я напишу, как они появятся")
			}
		} else {
			_, _ = b.tg.Send(&tb.User{ID: userID}, "Нет новых тарифных планов.")
		}
	} else {
		b.logger.Infof("Got a rate %s", rp.ToString())
		if u, exists := b.users[userID]; exists && u.AwaitsForNext {
			b.logger.Infof("Removing user from AwaitsForNext state")
			u.AwaitsForNext = false
			b.saveUsers(context.Background())
		}
		if wait {
			if u, exists := b.users[userID]; !exists || !u.Subscribed {
				b.logger.Infof("Got a rp %s after wait, but the user has unsubscribed", rp.ToString())
				b.moderator.Skip(rp)
				return
			}
		}
		b.sendRatePlan(userID, rp, count)
	}
}

func (b *moderBot) loadUsers(ctx context.Context) error {
	users, err := ytstorage.Load(ctx, reflect.TypeOf(user{}), b.cfg.BaseYTPath+"/users", b.cfg.YTToken, b.cfg.YTCluster)
	if err != nil {
		return err
	}
	loaded := map[int]*user{}
	for _, u := range users {
		us := u.(*user)
		if _, allowed := b.allowedLogins[strings.ToLower(us.Login)]; !allowed {
			b.logger.Infof("Disallowed user %s", us.Login)
			continue
		}
		loaded[us.UserID] = us
	}
	b.users = loaded
	for _, u := range loaded {
		if u.Subscribed && u.AwaitsForNext {
			b.logger.Infof("Will start blocking wait for user %v", u)
			go b.next(u.UserID, true)
		}
	}
	b.logger.Infof("Done loading users")
	return nil
}

func (b *moderBot) saveUsers(ctx context.Context) {
	var sCopy []*user
	for _, s := range b.users {
		sCopy = append(sCopy, s)
	}
	go func() {
		err := ytstorage.Save(ctx, sCopy, b.cfg.BaseYTPath+"/users", 1, b.cfg.YTToken, b.cfg.YTCluster)
		if err != nil {
			b.logger.Errorf("Unable to save users: %v", err)
		}
	}()
}

func (b *moderBot) loadChats(ctx context.Context) error {
	chats, err := ytstorage.Load(ctx, reflect.TypeOf(chat{}), b.cfg.BaseYTPath+"/chats", b.cfg.YTToken, b.cfg.YTCluster)
	if err != nil {
		return err
	}
	loaded := map[int64]*chat{}
	for _, c := range chats {
		ch := c.(*chat)
		loaded[ch.ID] = ch
	}
	b.alertChats = loaded
	return nil
}

func (b *moderBot) saveChats(ctx context.Context) {
	var cCopy []*chat
	for _, c := range b.alertChats {
		cCopy = append(cCopy, c)
	}
	go func() {
		err := ytstorage.Save(ctx, cCopy, b.cfg.BaseYTPath+"/chats", 1, b.cfg.YTToken, b.cfg.YTCluster)
		if err != nil {
			b.logger.Errorf("Unable to save chats: %v", err)
		}
	}()
}

func NewModerBot(moderator *moderator.Moderator, cfg *botCfg, logger log.Logger) (*moderBot, error) {
	settings := tb.Settings{
		Token:  cfg.TGToken,
		Poller: &tb.LongPoller{Timeout: 5 * time.Second},
	}
	tg, err := tb.NewBot(settings)
	if err != nil {
		return nil, err
	}

	allowedLoginsMap := map[string]bool{}
	for _, login := range cfg.AllowedLogins {
		allowedLoginsMap[strings.ToLower(login)] = true
	}
	bot := &moderBot{
		moderator:      moderator,
		tg:             tg,
		allowedLogins:  allowedLoginsMap,
		pendingReplies: map[int]string{},
		logger:         logger,
		cfg:            cfg,
	}
	tg.Handle("/start", bot.handleStart)
	tg.Handle("/stop", bot.handleStop)
	tg.Handle("/check", bot.handleCheck)
	tg.Handle("/get", bot.handleGet)
	tg.Handle("/list", bot.handleList)
	tg.Handle("/stats", bot.handleStats)
	tg.Handle("/startalerts", bot.handleStartAlerts)
	tg.Handle("/stopalerts", bot.handleStopAlerts)
	tg.Handle("\f"+showButtonSlug, bot.showButtonPressed)
	tg.Handle("\f"+hideButtonSlug, bot.hideButtonPressed)
	tg.Handle("\f"+moreButtonSlug, bot.moreButtonPressed)
	tg.Handle("\f"+cancellationTicketButtonSlug, bot.cancellationTicketPressed)
	tg.Handle("\f"+customTicketButtonSlug, bot.customTicketPressed)
	tg.Handle(tb.OnText, bot.replyReceived)
	moderator.HandleTicketCreation(bot.handleTicketsCreated)
	moderator.HandleAlert(bot.handleAlert)
	err = bot.loadUsers(context.Background())
	if err != nil {
		logger.Errorf("unable to load users table: %v", err)
		bot.users = map[int]*user{}
	}
	err = bot.loadChats(context.Background())
	if err != nil {
		logger.Errorf("unable to load alerts chat table: %v", err)
		bot.alertChats = map[int64]*chat{}
	}
	return bot, nil
}

func parseData(data string) (proto2.EPartnerId, string, string) {
	parts := strings.Split(data, ":")
	if len(parts) != 3 {
		return 0, "", ""
	}
	p, err := strconv.Atoi(parts[0])
	if err != nil {
		return 0, "", ""
	}
	return proto2.EPartnerId(p), parts[1], parts[2]
}

func getButtons(data string, selectedIndex int, showCancellationTicket bool, numInQueue int) [][]tb.InlineButton {
	show := tb.InlineButton{
		Unique: showButtonSlug,
		Text:   "Показываем",
		Data:   data,
	}

	hide := tb.InlineButton{
		Unique: hideButtonSlug,
		Text:   "Прячем",
		Data:   data,
	}

	ticket := tb.InlineButton{
		Unique: customTicketButtonSlug,
		Text:   "Тикет",
		Data:   data,
	}

	cancelTicket := tb.InlineButton{
		Unique: cancellationTicketButtonSlug,
		Text:   "Тикет о правилах отмены",
		Data:   data,
	}

	var moreText string
	if numInQueue > 0 {
		moreText = fmt.Sprintf("Ещё тариф (%d)", numInQueue)
	} else {
		moreText = "Ещё тариф"
	}

	more := tb.InlineButton{
		Unique: moreButtonSlug,
		Text:   moreText,
		Data:   data,
	}

	if selectedIndex == 0 {
		show.Text = "✔️ " + show.Text
	}
	if selectedIndex == 2 {
		hide.Text = "✔️ " + hide.Text
	}
	if showCancellationTicket {
		ticket.Text = "Другой тикет"
		return [][]tb.InlineButton{{show, hide}, {cancelTicket, ticket}, {more}}
	} else {
		return [][]tb.InlineButton{{show, hide}, {ticket}, {more}}
	}

}

func (b *moderBot) getItemDescription(item *moderator.RatePlanItem) string {
	var partnerName string
	switch item.PartnerID {
	case proto2.EPartnerId_PI_TRAVELLINE:
		partnerName = "Travelline"
	case proto2.EPartnerId_PI_BNOVO:
		partnerName = "BNovo"
	}
	message := fmt.Sprintf("<b>%s-отель %s (%s) — тарифный план '%s' (%s)</b>\r\n%s",
		partnerName, item.HotelName, item.OriginalID, item.Name, item.RatePlanID, item.Description)
	message = strings.ReplaceAll(message, "<p>", "")
	message = strings.ReplaceAll(message, "</p>", "")
	message = strings.ReplaceAll(message, "<ul>", "")
	message = strings.ReplaceAll(message, "</ul>", "")
	message = strings.ReplaceAll(message, "<li>", " - ")
	message = strings.ReplaceAll(message, "</li>", "")
	message = strings.ReplaceAll(message, "<span>", "")
	message = strings.ReplaceAll(message, "</span>", "")
	message = strings.ReplaceAll(message, "<br>", "\r\n")
	message = strings.ReplaceAll(message, "<br />", "\r\n")
	message = strings.ReplaceAll(message, "&nbsp;", " ")

	if item.CancellationYandex != "" || item.CancellationPartner != "" {
		message += fmt.Sprintf("\r\n"+
			"<b>Правила отмены текстом:</b> %s\r\n"+
			"<b>Правила отмены флагами:</b> %s", item.CancellationPartner, item.CancellationYandex)
	}
	if item.TokenExample != "" {
		message += fmt.Sprintf("\r\n\r\n<a href='%s?label=%s&token=%s'>Пример оффера</a>\r\n", b.cfg.LandingPrefix, b.cfg.LandingLabel, item.TokenExample)
	}
	return message
}
