package main

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/hotels/devops/partner_data_moderator/tg-bot/internal/moderator"
	"a.yandex-team.ru/travel/hotels/devops/partner_data_moderator/tg-bot/internal/partners"
	"a.yandex-team.ru/travel/hotels/lib/go/st"
	"a.yandex-team.ru/travel/hotels/lib/go/ytqueue"
	"context"
	"time"
)

func main() {
	config, err := LoadConfig(context.Background())
	if err != nil {
		panic(err)
	}

	var queues []ytqueue.QueueConfig
	for _, c := range config.YT.OfferClusters {
		q := ytqueue.QueueConfig{
			Proxy: c,
			Path:  config.YT.OfferBusPath,
			Token: config.YT.Token,
		}
		queues = append(queues, q)
	}

	zapCfg := zap.ConsoleConfig(log.InfoLevel)
	if config.LogDir != "" {
		zapCfg.OutputPaths = append(zapCfg.OutputPaths, config.LogDir+"/moderator.log")
		zapCfg.ErrorOutputPaths = append(zapCfg.ErrorOutputPaths, config.LogDir+"/moderator.log")
	}
	logger, err := zap.New(zapCfg)
	if err != nil {
		panic(err)
	}

	ctx := context.Background()

	cfg := &moderator.Config{
		YTToken:           config.YT.Token,
		RatePlanWLProxies: config.YT.WhiteListClusters,
		StorageProxy:      config.YT.MainCluster,
		StoragePath:       config.YT.Path + "/storage",
		TLPath:            config.YT.Path + "/travelline",
		BNPath:            config.YT.Path + "/bnovo",
		OfferQueues:       queues,
		QueueSize:         config.QueueSize,
		RepeatInterval:    time.Duration(config.RepeatIntervalMinutes) * time.Minute,
		SaveInterval:      time.Duration(config.SaveIntervalMinutes) * time.Minute,
		TicketInterval:    time.Duration(config.ST.TicketIntervalMinutes) * time.Minute,
		TLTrackerQueue:    config.ST.TLQueue,
		BNTrackerQueue:    config.ST.BNQueue,
		TLTags:            config.ST.TLTags,
		BNTags:            config.ST.BNTags,
		LoadWhiteLists:    config.LoadWhiteLists,
		TMPDir:            config.TMPDir,
		QueueAlertLimit:   config.QueueAlertLimit,
	}
	tlClient := partners.TravellineClient{
		BaseURL: config.Partners.Travelline.BaseURL,
		APIKey:  config.Partners.Travelline.APIKey,
	}
	bnClient := partners.BNovoClient{
		Username:   config.Partners.BNovo.Username,
		Password:   config.Partners.BNovo.Password,
		PrivateAPI: config.Partners.BNovo.PrivateURL,
		PublicAPI:  config.Partners.BNovo.PublicURL,
	}
	stClient := st.NewStartrekClientForOrg(config.ST.Token, config.ST.OrgID)

	m, err := moderator.NewModerator(ctx, cfg, tlClient, bnClient, stClient, logger)
	if err != nil {
		panic(err)
	}
	botCfg := &botCfg{
		TGToken:           config.Bot.Token,
		AllowedLogins:     config.Bot.AllowedLogins,
		BaseYTPath:        config.YT.Path,
		YTCluster:         config.YT.MainCluster,
		YTToken:           config.YT.Token,
		Night:             config.Bot.Night,
		RatePlanListLimit: config.Bot.RatePlanListLimit,
		LandingLabel:      config.LandingLabel,
		LandingPrefix:     config.LandingPrefix,
	}
	bot, err := NewModerBot(m, botCfg, logger)
	if err != nil {
		panic(err)
	}
	m.DumpPendingBackToQueue()
	bot.startListening()
}
