package moderator

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/hotels/devops/partner_data_moderator/tg-bot/internal/partners"
	"a.yandex-team.ru/travel/hotels/lib/go/st"
	"a.yandex-team.ru/travel/hotels/lib/go/ytqueue"
	"a.yandex-team.ru/travel/hotels/proto2"
	"bytes"
	"compress/zlib"
	"context"
	"encoding/gob"
	"fmt"
	"github.com/golang/protobuf/proto"
	"io/ioutil"
	"os"
	"strconv"
	"time"
)

type Config struct {
	YTToken           string
	RatePlanWLProxies []string
	StorageProxy      string
	StoragePath       string
	TLPath            string
	BNPath            string
	OfferQueues       []ytqueue.QueueConfig
	QueueSize         int
	RepeatInterval    time.Duration
	SaveInterval      time.Duration
	TicketInterval    time.Duration
	TLTrackerQueue    string
	BNTrackerQueue    string
	TLTags            []string
	BNTags            []string
	LoadWhiteLists    bool
	TMPDir            string
	QueueAlertLimit   int
}

type Moderator struct {
	cache          *ratePlanCache
	queue          chan *RatePlanItem
	commands       chan command
	changesPending bool
	tlClient       partners.TravellineClient
	bnClient       partners.BNovoClient
	stClient       *st.StartrekClient
	logger         log.Logger
	mainTicker     *time.Ticker
	ticketTicker   *time.Ticker
	counter        *DailyCounters
	ticketHandlers []TicketCreationHandler
	alertHandlers  []QueueAlertHandler
	cfg            *Config
	queueLength    int
}

type planRef struct {
	id    HotelID
	plan  string
	token string
}

type ticketResult struct {
	id     HotelID
	plan   string
	ticket string
	err    error
}

type DailyCounters struct {
	NewRatePlans       int
	ModeratedRatePlans int
	ModeratedByUser    map[string]int
	CreatedAt          time.Time
}

type Statistics struct {
	Counter        DailyCounters
	QueueSize      int
	InProgressSize int
}

type TicketCreationHandler func(username string, ticketKeys []string)

type QueueAlertHandler func(size int, alert bool)

type day struct {
	y int
	m time.Month
	d int
}

func (m *Moderator) Next(wait bool) (*RatePlanItem, int, error) {
	if !wait {
		m.logger.Infof("Will try to get new item from queue without waiting")
		defer m.logger.Infof("Done probing for item")
		select {
		case item, ok := <-m.queue:
			if !ok {
				return nil, 0, fmt.Errorf("input queue closed")
			}
			m.queueLength -= 1
			if item.State != Undecided {
				m.logger.Infof("Fetched an already moderated item %s, will fetch next", item.ToString())
				return m.Next(wait)
			}
			if !m.enrich(item) {
				m.logger.Infof("Removing item %s from cache", item.ToString())
				m.Remove(HotelID{OriginalID: item.OriginalID, PartnerID: item.PartnerID}, item.RatePlanID)
				return m.Next(wait)
			}
			item.ScheduleRequeueing(m, m.cfg.RepeatInterval)
			return item, m.queueLength, nil
		default:
			return nil, 0, nil
		}
	} else {
		m.logger.Infof("Will wait for new item to appear from queue")
		defer m.logger.Infof("Done waiting for item")
		item, ok := <-m.queue
		if !ok {
			return nil, 0, fmt.Errorf("input queue closed")
		}
		m.queueLength -= 1
		if item.State != Undecided {
			m.logger.Infof("Fetched an already moderated item %s, will fetch next", item.ToString())
			return m.Next(wait)
		}
		if !m.enrich(item) {
			m.logger.Infof("Removing item %s from cache", item.ToString())
			m.Remove(HotelID{OriginalID: item.OriginalID, PartnerID: item.PartnerID}, item.RatePlanID)
			return m.Next(wait)
		}
		item.ScheduleRequeueing(m, m.cfg.RepeatInterval)
		return item, m.Stats().QueueSize, nil
	}
}

func (m *Moderator) Put(item *RatePlanItem) {
	if item.State != Undecided {
		item.CancelRequeueing()
	}
	cmd := putCMD(item)
	m.commands <- cmd
	cmd.getResponse()
}

func (m *Moderator) Get(id HotelID, plan string, enrich bool) *RatePlanItem {
	cmd := getCMD(id, plan)
	m.logger.Infof("Fetching rp %s for hotel %v", plan, id)
	m.commands <- cmd
	resp := cmd.getResponse()
	if resp.items == nil {
		return nil
	} else {
		m.logger.Infof("Plain offer fetched")
		item := resp.items[0]
		if enrich {
			m.enrich(item)
		}
		return item
	}
}

func (m *Moderator) List(id HotelID, limit int, enrich bool) ([]*RatePlanItem, bool) {
	cmd := listCMD(id)
	m.commands <- cmd
	resp := cmd.getResponse()
	if resp.items == nil {
		return nil, false
	}
	items := resp.items
	cropped := false
	if len(resp.items) > limit {
		items = items[:limit]
		cropped = true
	}
	if enrich {
		for i := range items {
			m.enrich(resp.items[i])
		}
	}
	return items, cropped
}

func (m *Moderator) Remove(id HotelID, plan string) {
	cmd := removeCMD(id, plan)
	m.commands <- cmd
	cmd.getResponse()
}

func (m *Moderator) Skip(item *RatePlanItem) {
	if item.State == Undecided {
		item.CancelRequeueing()
		m.Requeue(item)
	} else {
		m.logger.Infof("Attempt to skip an already moderated item %s, no action taken", item.ToString())
	}
}

func (m *Moderator) Stats() *Statistics {
	cmd := statsCMD()
	m.commands <- cmd
	rsp := cmd.getResponse()
	return rsp.stats
}

func (m *Moderator) HandleTicketCreation(handler TicketCreationHandler) {
	m.ticketHandlers = append(m.ticketHandlers, handler)
}

func (m *Moderator) HandleAlert(handler QueueAlertHandler) {
	m.alertHandlers = append(m.alertHandlers, handler)
}

func NewModerator(ctx context.Context, cfg *Config, tlClient partners.TravellineClient, bnClient partners.BNovoClient,
	stClient *st.StartrekClient, logger log.Logger) (*Moderator, error) {
	cache, err := newRatePlanCache(ctx, cfg.StoragePath, cfg.TLPath, cfg.BNPath, cfg.YTToken, cfg.StorageProxy,
		cfg.RatePlanWLProxies, cfg.LoadWhiteLists, logger)
	if err != nil {
		return nil, fmt.Errorf("unable to initialize rateplan cache: %w", err)
	}
	queue := make(chan *RatePlanItem, cfg.QueueSize)
	commands := make(chan command)
	m := Moderator{
		cache:          cache,
		queue:          queue,
		commands:       commands,
		changesPending: false,
		tlClient:       tlClient,
		bnClient:       bnClient,
		logger:         logger,
		mainTicker:     time.NewTicker(cfg.SaveInterval),
		ticketTicker:   time.NewTicker(cfg.TicketInterval),
		stClient:       stClient,
		cfg:            cfg,
	}
	m.loadCounter()
	go m.run(ctx)
	return &m, nil
}

func (m *Moderator) run(ctx context.Context) {
	defer close(m.queue)
	offerChannel := ytqueue.Read(ctx, m.cfg.OfferQueues, m.logger)
	saveResChan := make(chan error)
	ticketResChan := make(chan []ticketResult)
	for {
		select {
		case <-ctx.Done():
			return
		case <-m.mainTicker.C:
			m.logger.Debug("Main timer ticks")
			if m.changesPending {
				m.logger.Info("Will save caches")
				m.cache.save(ctx, saveResChan)
			}
			m.checkMetrics()
		case <-m.ticketTicker.C:
			m.generateTickets(ctx, ticketResChan)
		case ticketResult := <-ticketResChan:
			ticketsByLogin := map[string]map[string]bool{}
			for _, r := range ticketResult {
				if r.err != nil {
					m.logger.Errorf("Error while generating ticket: %v", r.err)
				} else {
					item := m.cache.get(r.id, r.plan)
					if item != nil {
						m.logger.Infof("Created ticket %s for item %s", r.ticket, item.ToString())
						item.Comment = "" // здесь возможна гонка, если для тарифа написали новый комментарий
						item.Tickets = append(item.Tickets, r.ticket)
						item.VerifiedTimestamp = time.Now().Unix()
						m.cache.put(item)
						if _, e := ticketsByLogin[item.Moderator]; !e {
							ticketsByLogin[item.Moderator] = map[string]bool{}
						}
						ticketsByLogin[item.Moderator][r.ticket] = true
					}
				}
			}
			go func() {
				for login, tMap := range ticketsByLogin {
					var tickets []string
					for t := range tMap {
						tickets = append(tickets, t)
					}
					for _, h := range m.ticketHandlers {
						h(login, tickets)
					}
				}
			}()

		case saveResult := <-saveResChan:
			if saveResult != nil {
				m.logger.Errorf("unable to save caches: %s", saveResult)
			} else {
				m.changesPending = false
				m.logger.Info("Caches saved")
			}
		case cmd := <-m.commands:
			switch cmd.action {
			case put:
				changed := m.cache.put(cmd.item)
				if changed {
					m.logger.Infof("%s has put %s to cache", cmd.item.Moderator, cmd.item.ToString())
					m.counter.ModeratedRatePlans++
					c, found := m.counter.ModeratedByUser[cmd.item.Moderator]
					if !found {
						m.counter.ModeratedByUser[cmd.item.Moderator] = 1
					} else {
						m.counter.ModeratedByUser[cmd.item.Moderator] = c + 1
					}
					m.changesPending = true
				}
				cmd.respond(cmdResponse{})
			case get:
				item := m.cache.get(cmd.id, cmd.plan)
				if item == nil {
					cmd.respond(cmdResponse{items: nil})
				} else {
					cmd.respond(cmdResponse{items: []*RatePlanItem{item}})
				}
			case list:
				items := m.cache.list(cmd.id)
				cmd.respond(cmdResponse{items: items})
			case remove:
				changed := m.cache.remove(cmd.id, cmd.plan)
				if changed {
					m.logger.Infof("Removing item with key %s from cache", cmd.id)
					m.changesPending = true
				}
				cmd.respond(cmdResponse{})
			case stats:
				var counterCopy = *m.counter
				inProgress := m.countUndecided()
				stats := &Statistics{
					Counter:        counterCopy,
					QueueSize:      m.queueLength,
					InProgressSize: inProgress,
				}
				cmd.respond(cmdResponse{stats: stats})
			}
		case q := <-offerChannel:
			msg, err := decodeMessage(q.Bytes)
			if err != nil {
				m.logger.Errorf("unable to decode message: %v", err)
				continue
			}
			for _, ref := range getBoYPlans(msg, m.logger) {
				item := m.cache.get(ref.id, ref.plan)
				if item == nil {
					m.logger.Infof("Новый тарифный план %s для %s-отеля %s", ref.plan,
						ref.id.PartnerID.String(), ref.id.OriginalID)
					m.counter.NewRatePlans++
					item = &RatePlanItem{
						PartnerID:       ref.id.PartnerID,
						OriginalID:      ref.id.OriginalID,
						RatePlanID:      ref.plan,
						State:           Undecided,
						QueuedTimestamp: time.Now().Unix(),
						TokenExample:    ref.token,
					}
					m.cache.put(item)
					m.changesPending = true
					select {
					case m.queue <- item:
						m.queueLength += 1
						continue
					default:
						m.logger.Debugf("queue limit exceeded")
					}
				}
			}
		}
	}
}

func (m *Moderator) countUndecided() int {
	undecided := 0
	for _, h := range m.cache.cache {
		for _, i := range h {
			if i.State == Undecided {
				undecided++
			}
		}
	}
	return undecided
}

func (m *Moderator) enrich(item *RatePlanItem) bool {
	if item.PartnerID == proto2.EPartnerId_PI_TRAVELLINE {
		m.logger.Infof("Enriching with TL")
		rp, h, err := m.tlClient.GetTLRatePlanAndHotel(item.OriginalID, item.RatePlanID)
		if err != nil {
			m.logger.Errorf("unable to get tl rateplan/hotel info: %s", err)
			return false
		}
		if rp != nil {
			item.Name = rp.Name
			item.Description = rp.Description
		} else {
			m.logger.Warnf("no tl rateplan info")
			return false
		}
		if h != nil {
			item.HotelName = h.Name
		} else {
			m.logger.Warnf("no tl hotel info")
			return false
		}
	}
	if item.PartnerID == proto2.EPartnerId_PI_BNOVO {
		m.logger.Infof("Enriching with BN")
		rp, err := m.bnClient.GetPlan(item.OriginalID, item.RatePlanID)
		if err != nil {
			m.logger.Errorf("unable to get bn rateplan info: %s", err)
			return false
		}
		h, err := m.bnClient.GetAccount(item.OriginalID)
		if err != nil {
			m.logger.Errorf("unable to get bn hotel info: %s", err)
			return false
		}
		if rp != nil {
			item.Name = rp.GetName()
			item.Description = rp.GetDescription()
			item.CancellationPartner = rp.GetCancellationRules()
			item.CancellationYandex = rp.GetCancellationRulesByFlags()
		} else {
			m.logger.Warnf("no bn rateplan info")
			return false
		}
		if h != nil {
			item.HotelName = h.Name
		} else {
			m.logger.Warnf("no bn hotel info")
			return false
		}
	}
	m.logger.Infof("Done enriching")
	return true
}

func (m *Moderator) generateTickets(ctx context.Context, resChan chan<- []ticketResult) bool {
	itemsWithComments := map[HotelID][]*RatePlanItem{}
	num := 0
	numTick := 0
	for hotelID, h := range m.cache.cache {
		has := false
		for _, i := range h {
			if i.Comment != "" {
				cp := *i
				m.enrich(&cp)
				num++
				has = true
				itemsWithComments[hotelID] = append(itemsWithComments[hotelID], &cp)
			}
		}
		if has {
			numTick++
		}
	}
	if num == 0 {
		return false
	} else {
		m.logger.Infof("Will generate %d tickets for %d rateplans", numTick, num)
	}

	go func() {
		var res []ticketResult
		for id, items := range itemsWithComments {
			hotelName := items[0].HotelName
			header := fmt.Sprintf("Проблемы с тарифными планами для отеля %s (id=%s)", hotelName, id.OriginalID)
			body := fmt.Sprintf("Для отеля %s (%s) обнаружены проблемы со следующими тарифными планами:\n\n",
				id.OriginalID, hotelName)
			var okItems []*RatePlanItem
			for _, item := range items {
				reqFields := []string{hotelName, item.Name, item.Moderator}
				missing := false
				for _, f := range reqFields {
					if f == "" {
						missing = true
						break
					}
				}
				if missing {
					res = append(res, ticketResult{
						id:   id,
						plan: item.RatePlanID,
						err:  fmt.Errorf("some of required data for '%s' is missing", item.ToString()),
					})
					continue
				} else {
					okItems = append(okItems, item)
				}
				decision := ""
				if item.State == Hide {
					decision = " — **продажи отключены!**"
				}
				cancellationDesc := ""
				if item.CancellationYandex != "" {
					cancellationDesc = fmt.Sprintf("\n"+
						"__Правила отмены текстом__: %s\n"+
						"__Правила отмены флагами__: %s\n", item.CancellationPartner, item.CancellationYandex)
				}
				line := fmt.Sprintf(
					"**Тарифный план %s (%s)**%s\n%s"+
						"__Описание__: %s\n\n"+
						"__Примечание от %s__: %s\n\n", item.RatePlanID, item.Name, decision, cancellationDesc, item.Description, item.Moderator, item.Comment)
				body += line
			}
			if len(okItems) > 0 {
				var queueName string
				var tags []string
				if items[0].PartnerID == proto2.EPartnerId_PI_BNOVO {
					queueName = m.cfg.BNTrackerQueue
					tags = m.cfg.BNTags
				}
				if items[0].PartnerID == proto2.EPartnerId_PI_TRAVELLINE {
					queueName = m.cfg.TLTrackerQueue
					tags = m.cfg.TLTags
				}

				req := &st.CreateIssueRequest{}
				req.Body.Queue = queueName
				req.Body.Summary = header
				req.Body.Description = body
				req.Body.Tags = tags
				response, err := req.Execute(m.stClient, ctx)
				if err != nil {
					for _, i := range okItems {
						res = append(res, ticketResult{
							id:   id,
							plan: i.RatePlanID,
							err:  err,
						})
					}
				} else {
					for _, i := range okItems {
						res = append(res, ticketResult{
							id:     id,
							plan:   i.RatePlanID,
							ticket: response.Issue.Ref.Key,
						})
					}
				}
			}
		}
		m.logger.Infof("Done generating tickets")
		resChan <- res
	}()
	return true
}

func (m *Moderator) DumpPendingBackToQueue() {
	for _, h := range m.cache.cache {
		for _, i := range h {
			m.Requeue(i)
		}
	}
}

func (m *Moderator) Requeue(i *RatePlanItem) {
	now := time.Now()
	if i.State == Undecided {
		i.QueuedTimestamp = now.Unix()
		m.changesPending = true
		select {
		case m.queue <- i:
			m.logger.Infof("Item %s requeued", i.ToString())
			m.queueLength += 1
			break
		default:
			m.logger.Warnf("Item %s skipped, as the queue is full", i.ToString())
			break
		}
	}
}

func (m *Moderator) checkMetrics() {
	today := dayFromTime(time.Now())
	c := dayFromTime(m.counter.CreatedAt)
	if today != c {
		m.logger.Infof("Rotating metrics as the new day has began")
		m.counter = newCounter()
	}
	size := m.countUndecided()
	if size >= m.cfg.QueueAlertLimit {
		go func() {
			for _, h := range m.alertHandlers {
				h(size, true)
			}
		}()
	} else {
		go func() {
			for _, h := range m.alertHandlers {
				h(size, false)
			}
		}()
	}
	m.saveCounter()
}

func dayFromTime(t time.Time) day {
	y, m, d := t.Date()
	return day{
		y: y,
		m: m,
		d: d,
	}
}

func decodeMessage(data []byte) (*proto2.TSearcherMessage, error) {
	b := bytes.NewReader(data)
	z, err := zlib.NewReader(b)
	if err != nil {
		return nil, err
	}
	defer func() { _ = z.Close() }()
	p, err := ioutil.ReadAll(z)
	if err != nil {
		return nil, err
	}
	var msg proto2.TSearcherMessage
	err = proto.Unmarshal(p, &msg)
	if err != nil {
		return nil, err
	}
	return &msg, nil
}

func newCounter() *DailyCounters {
	c := DailyCounters{
		NewRatePlans:       0,
		ModeratedRatePlans: 0,
		ModeratedByUser:    map[string]int{},
		CreatedAt:          time.Now(),
	}
	return &c
}

func (m *Moderator) loadCounter() {
	file, err := os.Open(m.cfg.TMPDir + "/counters.gob")
	if err != nil {
		m.logger.Errorf("Unable to load counters file: %v", err)
		m.counter = newCounter()
		return
	}
	defer file.Close()
	counter := newCounter()
	err = gob.NewDecoder(file).Decode(counter)
	if err != nil {
		m.logger.Errorf("Unable to load counters file: %v", err)
		m.counter = newCounter()
		return
	}
	m.counter = counter
}

func (m *Moderator) saveCounter() {
	file, err := os.Create(m.cfg.TMPDir + "/counters.gob")
	if err != nil {
		m.logger.Errorf("Unable to save counters file: %v", err)
		return
	}
	defer file.Close()
	err = gob.NewEncoder(file).Encode(m.counter)
	if err != nil {
		m.logger.Errorf("Unable to save counters file: %v", err)
	}
}

func getBoYPlans(msg *proto2.TSearcherMessage, logger log.Logger) []planRef {
	var result []planRef

	if msg.Response != nil && msg.Response.GetOffers() != nil {
		for _, r := range msg.Response.GetOffers().Offer {
			var rp string
			switch msg.Request.HotelId.PartnerId {
			case proto2.EPartnerId_PI_BNOVO:
				if r.PartnerSpecificData == nil {
					logger.Warnf("No partner-specific-data for BN offer")
					continue
				}
				rp = strconv.FormatInt(r.PartnerSpecificData.GetBNovoData().RatePlanId, 10)
			case proto2.EPartnerId_PI_TRAVELLINE:
				if r.PartnerSpecificData == nil {
					logger.Warnf("No partner-specific-data for BN offer")
					continue
				}
				rp = r.PartnerSpecificData.GetTravellineData().RatePlanCode
			default:
				continue
			}
			i := planRef{
				id: HotelID{
					PartnerID:  msg.Request.HotelId.PartnerId,
					OriginalID: msg.Request.HotelId.OriginalId,
				},
				plan:  rp,
				token: r.LandingInfo.GetLandingTravelToken(),
			}
			result = append(result, i)
		}
	}
	return result
}
