package moderator

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/hotels/lib/go/ytstorage"
	"a.yandex-team.ru/travel/hotels/proto2"
	"context"
	"fmt"
	"reflect"
	"strconv"
	"sync"
	"time"
)

type RatePlanState int

const (
	Undecided RatePlanState = iota
	Show      RatePlanState = iota
	Hide      RatePlanState = iota
	Warn      RatePlanState = iota
)

type RatePlanItem struct {
	PartnerID           proto2.EPartnerId `yson:"PartnerId"`
	OriginalID          string            `yson:"OriginalId"`
	RatePlanID          string            `yson:"RatePlanId"`
	State               RatePlanState     `yson:"State"`
	HotelName           string            `yson:"HotelName"`
	Name                string            `yson:"Name"`
	Description         string            `yson:"Description"`
	CancellationYandex  string            `yson:"CancellationYandex"`
	CancellationPartner string            `yson:"CancellationPartner"`
	Moderator           string            `yson:"Moderator"`
	VerifiedTimestamp   int64             `yson:"VerifiedTimestamp"`
	QueuedTimestamp     int64             `yson:"QueuedTimestamp"`
	Comment             string            `yson:"Comment"`
	Tickets             []string          `yson:"Tickets"`
	TokenExample        string            `yson:"TokenExample"`

	requeueingCancellation context.CancelFunc
}

func (i *RatePlanItem) ToString() string {
	var prefix string
	var state string
	switch i.PartnerID {
	case proto2.EPartnerId_PI_BNOVO:
		prefix = "bn"
	case proto2.EPartnerId_PI_TRAVELLINE:
		prefix = "tl"
	}
	switch i.State {
	case Show:
		state = "show"
	case Hide:
		state = "hide"
	case Warn:
		state = "warn"
	case Undecided:
		state = "pending"
	}
	hotelName := i.HotelName
	if hotelName == "" {
		hotelName = "unknown"
	}
	rpName := i.Name
	if rpName == "" {
		rpName = "unknown"
	}

	return fmt.Sprintf("[(%s:%s \"%s\") - %s (%s): %s]", prefix, i.OriginalID, hotelName, i.RatePlanID, rpName, state)
}

func (i *RatePlanItem) ScheduleRequeueing(moderator *Moderator, duration time.Duration) {
	timer := time.After(duration)
	ctx, cancel := context.WithCancel(context.Background())
	i.requeueingCancellation = cancel
	go func() {
		select {
		case <-timer:
			moderator.Requeue(i)
			break
		case <-ctx.Done():
			break
		}
	}()
}

func (i *RatePlanItem) CancelRequeueing() {
	if i.requeueingCancellation != nil {
		i.requeueingCancellation()
		i.requeueingCancellation = nil
	}
}

type tlItem struct {
	HotelCode    string `yson:"HotelCode"`
	RatePlanCode string `yson:"RatePlanCode"`
	Enabled      bool   `yson:"Enabled"`
	Warn         bool   `yson:"Warn"`
}

type bnItem struct {
	AccountID  int64 `yson:"AccountId"`
	RatePlanID int64 `yson:"RatePlanId"`
	Enabled    bool  `yson:"Enabled"`
	Warn       bool  `yson:"Warn"`
}

type HotelID struct {
	PartnerID  proto2.EPartnerId
	OriginalID string
}

type ratePlanCache struct {
	cache     map[HotelID]map[string]*RatePlanItem
	ytToken   string
	mainProxy string
	wlProxies []string
	mainPath  string
	tlPath    string
	bnPath    string
}

func (r *ratePlanCache) put(item *RatePlanItem) bool {
	hid := HotelID{
		PartnerID:  item.PartnerID,
		OriginalID: item.OriginalID,
	}
	hMap, exists := r.cache[hid]
	if !exists {
		hMap = map[string]*RatePlanItem{}
		r.cache[hid] = hMap
	}
	existing, rpExists := hMap[item.RatePlanID]
	if rpExists {
		if existing.VerifiedTimestamp != 0 && (item.VerifiedTimestamp == 0 || existing.VerifiedTimestamp > item.VerifiedTimestamp) {
			return false
		}
	}
	hMap[item.RatePlanID] = item
	return true
}

func (r *ratePlanCache) get(hotelID HotelID, planID string) *RatePlanItem {
	hMap, exists := r.cache[hotelID]
	if !exists {
		return nil
	}
	rp, rpExists := hMap[planID]
	if !rpExists {
		return nil
	}
	return rp
}

func (r *ratePlanCache) list(hotelID HotelID) []*RatePlanItem {
	hMap, exists := r.cache[hotelID]
	if !exists {
		return nil
	}
	res := make([]*RatePlanItem, len(hMap))
	i := 0
	for _, item := range hMap {
		res[i] = item
		i++
	}
	return res
}

func (r *ratePlanCache) remove(hotelID HotelID, planID string) bool {
	hMap, exists := r.cache[hotelID]
	if !exists {
		return false
	}
	_, rpExists := hMap[planID]
	if !rpExists {
		return false
	}
	delete(hMap, planID)
	return true
}

func (r *ratePlanCache) loadBN(ctx context.Context) error {
	items, err := ytstorage.Load(ctx, reflect.TypeOf(bnItem{}), r.bnPath+"-old", r.ytToken, r.mainProxy)
	if err != nil {
		return err
	}
	for _, i := range items {
		item := i.(*bnItem)
		var state RatePlanState
		if item.Enabled {
			state = Show
		} else {
			state = Hide
		}
		res := RatePlanItem{
			PartnerID:         proto2.EPartnerId_PI_BNOVO,
			OriginalID:        strconv.FormatInt(item.AccountID, 10),
			RatePlanID:        strconv.FormatInt(item.RatePlanID, 10),
			State:             state,
			Description:       "",
			Moderator:         "unknown",
			VerifiedTimestamp: time.Now().Unix(),
		}
		r.put(&res)
	}
	return nil
}

func (r *ratePlanCache) loadTL(ctx context.Context) error {
	items, err := ytstorage.Load(ctx, reflect.TypeOf(tlItem{}), r.tlPath+"-old", r.ytToken, r.mainProxy)
	if err != nil {
		return err
	}
	for _, i := range items {
		item := i.(*tlItem)
		var state RatePlanState
		if item.Enabled {
			state = Show
		} else {
			state = Hide
		}
		res := RatePlanItem{
			PartnerID:         proto2.EPartnerId_PI_TRAVELLINE,
			OriginalID:        item.HotelCode,
			RatePlanID:        item.RatePlanCode,
			State:             state,
			Description:       "",
			Moderator:         "unknown",
			VerifiedTimestamp: 1000,
		}
		r.put(&res)
	}
	return nil
}

func (r *ratePlanCache) load(ctx context.Context) error {
	items, err := ytstorage.Load(ctx, reflect.TypeOf(RatePlanItem{}), r.mainPath, r.ytToken, r.mainProxy)
	if err != nil {
		return err
	}
	for _, i := range items {
		item := i.(*RatePlanItem)
		r.put(item)
	}
	return nil
}

func (r *ratePlanCache) save(ctx context.Context, resChannel chan<- error) {
	var cacheCopy []RatePlanItem
	var tlItems []*tlItem
	var bnItems []*bnItem
	for _, m := range r.cache {
		for _, i := range m {
			cacheCopy = append(cacheCopy, *i)
			if i.PartnerID == proto2.EPartnerId_PI_TRAVELLINE && i.State != Undecided {
				tlItems = append(tlItems, &tlItem{
					HotelCode:    i.OriginalID,
					RatePlanCode: i.RatePlanID,
					Enabled:      i.State == Show || i.State == Warn,
					Warn:         i.State == Warn,
				})
			}
			if i.PartnerID == proto2.EPartnerId_PI_BNOVO && i.State != Undecided {
				accountID, _ := strconv.ParseInt(i.OriginalID, 10, 64)
				rpID, _ := strconv.ParseInt(i.RatePlanID, 10, 64)
				bnItems = append(bnItems, &bnItem{
					AccountID:  accountID,
					RatePlanID: rpID,
					Enabled:    i.State == Show || i.State == Warn,
					Warn:       i.State == Warn,
				})
			}
		}
	}
	go func() {
		var mainSave error
		var tlSave error
		var bnSave error
		wg := sync.WaitGroup{}
		wg.Add(3)
		go func() {
			defer wg.Done()
			mainSave = ytstorage.Save(ctx, cacheCopy, r.mainPath, 1, r.ytToken, r.mainProxy)
		}()
		go func() {
			defer wg.Done()
			tlSave = ytstorage.Save(ctx, tlItems, r.tlPath, 1, r.ytToken, r.wlProxies...)
		}()
		go func() {
			defer wg.Done()
			bnSave = ytstorage.Save(ctx, bnItems, r.bnPath, 1, r.ytToken, r.wlProxies...)
		}()
		wg.Wait()
		if mainSave != nil || tlSave != nil || bnSave != nil {
			err := fmt.Errorf("save failed. main=%v, bn=%v, tl=%v", mainSave, tlSave, bnSave)
			select {
			case resChannel <- err:
				return
			case <-ctx.Done():
				return
			}
		} else {
			select {
			case resChannel <- nil:
				return
			case <-ctx.Done():
				return
			}
		}
	}()

}

func newRatePlanCache(ctx context.Context, pathToStorage string, pathToTL string, pathToBN string, ytToken string,
	primaryProxy string, wlProxies []string, loadWL bool, logger log.Logger) (*ratePlanCache, error) {
	r := ratePlanCache{
		cache:     map[HotelID]map[string]*RatePlanItem{},
		ytToken:   ytToken,
		mainProxy: primaryProxy,
		wlProxies: wlProxies,
		mainPath:  pathToStorage,
		tlPath:    pathToTL,
		bnPath:    pathToBN,
	}

	err := r.load(ctx)
	if err != nil {
		logger.Warnf("unable to load rateplan cache: %v", err)
	}
	if loadWL {
		if pathToTL != "" {
			err = r.loadTL(ctx)
			if err != nil {
				logger.Warnf("unable to load TL rateplan list: %v", err)
			}
		}
		if pathToBN != "" {
			err = r.loadBN(ctx)
			if err != nil {
				logger.Warnf("unable to load BN rateplan list: %v", err)
			}
		}
		errChan := make(chan error)
		r.save(ctx, errChan)
		err = <-errChan
		if err != nil {
			logger.Warnf("unable to save rateplan cache: %v", err)
		}
	}
	return &r, nil
}
