package partners

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"
	"time"
)

type authResponse struct {
	Token string `json:"token"`
}

type accountResponse struct {
	Account Account `json:"account"`
}

type ratePlanResponse struct {
	Plans []Plan `json:"plans"`
}

type Plan struct {
	ID                     int64  `json:"id"`
	Name                   string `json:"name"`
	NameRu                 string `json:"name_ru"`
	Description            string `json:"description"`
	DescriptionRu          string `json:"description_ru"`
	CancellationRules      string `json:"cancellation_rules"`
	CancellationRulesRu    string `json:"cancellation_rules_ru"`
	CancellationDeadline   string `json:"cancellation_deadline"`
	CancellationFineType   string `json:"cancellation_fine_type"`
	CancellationFineAmount string `json:"cancellation_fine_amount"`
}

func (p Plan) GetName() string {
	return coalesce(p.NameRu, p.Name)
}

func (p Plan) GetDescription() string {
	return coalesce(p.DescriptionRu, p.Description)
}

func (p Plan) GetCancellationRules() string {
	return coalesce(p.CancellationRulesRu, p.CancellationRules)
}

func (p Plan) GetCancellationRulesByFlags() string {
	if p.CancellationDeadline == "0" {
		return "Бесплатная отмена"
	} else {
		rule := fmt.Sprintf("Бесплатная отмена за %s дней до заезда, потом штраф ", p.CancellationDeadline)
		switch p.CancellationFineType {
		case "2":
			rule += fmt.Sprintf("%s%%", p.CancellationFineAmount)
		case "3":
			rule += fmt.Sprintf("%s руб", p.CancellationFineAmount)
		case "4":
			rule += "первая ночь"
		case "5":
			rule += "вся сумма"
		}
		return rule
	}
}

type Account struct {
	Name string `json:"name"`
}

type BNovoClient struct {
	Username       string
	Password       string
	PrivateAPI     string
	PublicAPI      string
	token          string
	tokenUpdatedAt time.Time
}

func (c *BNovoClient) GetPlan(accountID string, planID string) (*Plan, error) {
	targetURL := fmt.Sprintf("%s/plans?account_id=%s", c.PublicAPI, accountID)
	resp, err := http.Get(targetURL)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("unexpected status code %d when calling BN API", resp.StatusCode)
	}
	result, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	var response ratePlanResponse
	err = json.Unmarshal(result, &response)
	if err != nil {
		return nil, err
	}
	for _, p := range response.Plans {
		id, _ := strconv.ParseInt(planID, 10, 64)
		if p.ID == id {
			return &p, nil
		}
	}
	return nil, nil
}

func (c *BNovoClient) GetAccount(accountID string) (*Account, error) {
	token, err := c.getToken()
	if err != nil {
		return nil, err
	}
	targetURL := fmt.Sprintf("%s/accounts/%s?token=%s", c.PrivateAPI, accountID, token)
	resp, err := http.Get(targetURL)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("unexpected status code %d when calling BN API", resp.StatusCode)
	}
	result, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	var response accountResponse
	err = json.Unmarshal(result, &response)
	if err != nil {
		return nil, err
	}
	return &response.Account, nil
}

func (c *BNovoClient) getToken() (string, error) {
	if c.token == "" || c.tokenUpdatedAt.Before(time.Now().Add(-1*time.Hour)) {
		newToken, err := c.authorize()
		if err != nil {
			return "", fmt.Errorf("unable to authorize: %w", err)
		}
		c.token = newToken
		c.tokenUpdatedAt = time.Now()
	}
	return c.token, nil
}

func (c *BNovoClient) authorize() (string, error) {
	data := url.Values{}
	data.Add("username", c.Username)
	data.Add("password", c.Password)
	targetURL := c.PrivateAPI + "/auth"
	resp, err := http.PostForm(targetURL, data)
	if err != nil {
		return "", err
	}
	if resp.StatusCode != 200 {
		return "", fmt.Errorf("unexpected status code %d when calling BN auth API", resp.StatusCode)
	}
	result, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return "", err
	}
	var response authResponse
	err = json.Unmarshal(result, &response)
	if err != nil {
		return "", err
	}
	return response.Token, nil
}

func coalesce(input ...string) string {
	for _, r := range input {
		if r != "" {
			return r
		}
	}
	return ""
}
