#! /usr/bin/env python
# -*- coding: utf-8 -*-

from gevent import monkey
monkey.patch_all()

import argparse
import getpass
import logging

import travel.library.python.tools as tools

from travel.hotels.devops.sandbox_planner import environments
from travel.hotels.devops.sandbox_planner.runner import Runner
from travel.hotels.devops.sandbox_planner.slack_notifier import SlackNotifier


def handle_update_plan(args):
    print("Command 'update-plan' was removed.")
    print("If you need to update plan, do following:")
    print(" 1. Run ./sandbox-planner validate-plan")
    print(" 2. Commit your changes to VCS and wait for automatics")


def handle_update_plan_internal(args):
    if args.yav_token:
        raise Exception("Never call it locally!")
    if not args.yav_token_testing and not args.yav_token_prod:
        raise Exception("Never call it locally!")
    notifier = SlackNotifier(args.notify_base_url, "SandboxPlanner - Plan", args.sandbox_task_id, args.commit_revision)
    notifier.register_build(args.commit_message, args.commit_author)
    if args.yav_token_testing:
        r = Runner(args.yav_token_testing, 'testing', args.yt_proxy, args.plan)
        r.handle_update_plan_internal(notifier, 'Testing', args.commit_revision, args.allow_downgrade)
    if args.yav_token_prod:
        r = Runner(args.yav_token_prod, 'prod', args.yt_proxy, args.plan)
        r.handle_update_plan_internal(notifier, 'Stable', args.commit_revision, args.allow_downgrade)


def handle_validate_plan(args):
    for env in environments.ENVIRONMENTS.keys():
        logging.info(f"Validating for env {env}")
        r = Runner(args.yav_token, env, args.yt_proxy, args.plan, create_clients=not args.no_check_delegation)
        r.handle_validate_plan(check_delegation=not args.no_check_delegation)


def handle_render_plan(args):
    r = Runner(args.yav_token, args.env, args.yt_proxy, args.plan)
    r.handle_render_plan()


def handle_prepare_dashboard(args):
    r = Runner(args.yav_token, args.env, args.yt_proxy, args.plan)
    r.handle_prepare_dashboard(args.col_count, args.dashboard_path)


def handle_regular_run(args):
    r = Runner(args.yav_token, args.env, args.yt_proxy, args.plan)
    r.handle_regular_run(args.plan_from, args.plan_to)


def handle_run_now(args):
    r = Runner(args.yav_token, args.env, args.yt_proxy, args.plan)
    r.handle_run_now(args.PLAN_ID, args.yt_trigger_path, args.notify_email)


def handle_drop_old_schedulers(args):
    r = Runner(args.yav_token, args.env, args.yt_proxy, args.plan)
    r.handle_drop_old_schedulers()


def main():
    FORMAT = '%(asctime)-15s | %(levelname)-4.4s | %(name)-12.12s | %(message)s'
    logging.basicConfig(level=logging.INFO, format=FORMAT)
    logging.getLogger("yt.packages.urllib3.connectionpool").setLevel(logging.ERROR)

    parser = argparse.ArgumentParser()

    parser.add_argument('--yav-token', help='OAuth token YaVault API')

    # Общие параметры
    parser.add_argument('--yt-proxy', nargs='+', default=['hahn', 'arnold'])
    parser.add_argument('--env', '-e', default='testing', choices=environments.ENVIRONMENTS)
    parser.add_argument('--plan')

    subparsers = parser.add_subparsers(dest='cmd')
    subparsers.required = True

    subparsers.add_parser("update-plan").set_defaults(func=handle_update_plan)

    update_plan_internal_parser = subparsers.add_parser("update-plan-internal")
    update_plan_internal_parser.set_defaults(func=handle_update_plan_internal)
    update_plan_internal_parser.add_argument('--yav-token-testing')
    update_plan_internal_parser.add_argument('--yav-token-prod')
    update_plan_internal_parser.add_argument('--notify-base-url', default='http://travel-slack-forwarder.yandex.net:9098/build-notifications')
    update_plan_internal_parser.add_argument('--commit-revision', type=int)
    update_plan_internal_parser.add_argument('--commit-author', default='')
    update_plan_internal_parser.add_argument('--commit-message', default='')
    update_plan_internal_parser.add_argument('--sandbox-task-id', default=0)
    update_plan_internal_parser.add_argument('--allow-downgrade', action='store_true')

    validate_plan_parser = subparsers.add_parser("validate-plan")
    validate_plan_parser.set_defaults(func=handle_validate_plan)
    validate_plan_parser.add_argument('--no-check-delegation', action='store_true')

    subparsers.add_parser("render-plan").set_defaults(func=handle_render_plan)

    prepare_dashboard_parser = subparsers.add_parser("prepare-dashboard")
    prepare_dashboard_parser.set_defaults(func=handle_prepare_dashboard)
    prepare_dashboard_parser.add_argument('--col-count', type=int, default=5)
    prepare_dashboard_parser.add_argument('--dashboard-path')

    regular_run_parser = subparsers.add_parser("regular-run")
    regular_run_parser.set_defaults(func=handle_regular_run)
    regular_run_parser.add_argument('--plan-from', default=15, type=int)
    regular_run_parser.add_argument('--plan-to', default=90, type=int)

    run_now_parser = subparsers.add_parser('run-now')
    run_now_parser.set_defaults(func=handle_run_now)
    run_now_parser.add_argument('PLAN_ID')
    run_now_parser.add_argument('--yt-trigger-path')
    run_now_parser.add_argument('--notify-email', default=getpass.getuser() + '@yandex-team.ru')

    drop_old_schedulers_parser = subparsers.add_parser('drop-old-schedulers')
    drop_old_schedulers_parser.set_defaults(func=handle_drop_old_schedulers)

    args = parser.parse_args(args=tools.replace_args_from_env())
    args.func(args)


if __name__ == '__main__':
    main()
