import time
from collections import OrderedDict
from textwrap import dedent
from typing import Dict, List, Optional
from pathlib import Path
import datetime
import gevent
import itertools
import logging
import re
import traceback

from croniter import croniter
from sandbox.common import rest
from sandbox.common import auth
import dateutil.parser
import jinja2
import yaml

from library.python import vault_client

from travel.hotels.devops.sandbox_planner import environments
from travel.hotels.devops.sandbox_planner.data import PlanFile, PlanItem, RawPlan, YtTriggerConfig
from travel.hotels.devops.sandbox_planner.yt_trigger import YtTrigger, YtTriggerPath
from travel.hotels.lib.python3.yt import ytlib
from travel.library.python.arcadia import arcadia_tools
from travel.library.python.time_interval import TimeInterval
import travel.hotels.devops.sandbox_planner.sb_tools as sb_tools


'''
1. Расписание:
  * id
  * cron-expr
  * command
  * aux options (concurrency,...)
2. Параметры: интервал планирования (от и до), env
3. алгоритм:
   Планируем интервал планирования - для каждой задачи - все запуски в пределах времени [now + plan_from, now + plan_to]

   Получаем список того, что надо запланировать:
     на каждый запуск - свой отдельный sb-scheduler
   Из sb получаем список шедулеров (по тэгу)
   Сверяем, применяем разницу
TODO
 * шаблонизация, например описать CPA один раз, запускать дважды (hahn+arnold) - надо еще условных переменных досыпать
 * разделить тэги planner-а и реальных задач
 * Проверять, что релиз ресурса в прод был слишком давно (!!) при заливке плана
'''

DEFAULT_TASK_TYPE = 'TRAVEL_RUN_BINARY'

TEMPLATABLE_PLAN_ITEM_REGEX = re.compile('{{([^}]+)}}')

YT_TRIGGER_PATH_PLACEHOLDER = '%YT_TRIGGER_PATH%'

DASHBOARD_TEMPLATE = '''
    {%- raw -%}
    # This file was generated by travel/hotels/devops/sandbox_planner
    # Do not try to edit manually

    name: Sandbox binary tasks ({{cluster}})
    parameters:
      service: sandbox
      cluster: '*'
    {% endraw -%}
    rows:
    {% for row in grid -%}
    - panels:
      {% for item in row -%}
      - title: '{{ item }}'
        graph: travel-sandbox-run-status
        selectors:
          plan_id: {{ item }}
      {% endfor %}
    {% endfor %}
'''


class Runner:
    TZ = datetime.timezone(datetime.timedelta(hours=3), 'MSK')
    YT_REVISION_ATTRIBUTE = '@_arcadia_revision'
    YT_TRIGGER_LIMIT = 50

    def __init__(self, yav_token, env, yt_proxies, plan_path, create_clients=True):
        self.plan_path = plan_path or arcadia_tools.get_arcadia_path(
            'travel', 'hotels', 'devops', 'sandbox_planner', 'plan'
        )
        self.yt_proxies = yt_proxies
        self.env = env
        self.environment = environments.ENVIRONMENTS[env]
        self.yt_clients = {px: None for px in yt_proxies}
        self.sb_client = None

        if not create_clients:
            return

        if not yav_token:
            logging.info("Have no YAV token, will try to use SSH key")

        yav_client = vault_client.instances.Production(authorization=yav_token)
        sb_token = self.resolve_secret(yav_client, self.environment.sb_token_secret)
        logging.info("Got %s Sandbox token from Vault" % self.env)
        self.sb_client = rest.Client(auth=auth.OAuth(sb_token))

        yt_token = self.resolve_secret(yav_client, self.environment.yt_token_secret)
        logging.info("Got %s YT token from Vault" % self.env)

        for px in yt_proxies:
            self.yt_clients[px] = ytlib.create_client(proxy=px, config={'token': yt_token})

    @staticmethod
    def parse_secret(secret):  # -> (secret_id, version, key)
        parts = secret.split('.')
        if len(parts) > 3:
            raise RuntimeError("Too much parts in secret %s" % secret)
        version = None
        key = None
        for part in parts[1:]:
            if part.startswith('ver-'):
                if version is not None:
                    raise RuntimeError("Duplicate version in secret %s" % secret)
                version = part
            elif part.startswith('key-'):
                if key is not None:
                    raise RuntimeError("Duplicate key in secret %s" % secret)
                key = part[4:]
            else:
                raise RuntimeError("Unknown part in secret %s" % secret)
        return parts[0], version, key

    @staticmethod
    def resolve_secret(yav_client, secret):
        secret_id, version, key = Runner.parse_secret(secret)
        data = yav_client.get_version(version or secret_id)['value']
        if key is None:
            if len(data) == 1:
                return list(data.values())[0]
            else:
                raise RuntimeError("Secret %s has multiple keys" % secret)
        return data[key]

    @staticmethod
    def pop_other_envs(env, item, key):
        for other_env in environments.ENVIRONMENTS:
            if other_env != env:
                item.pop(key + '_' + other_env, None)

    def pop_plan_entry_parameter(self, env, plan_entry, key, default=None, joiner=None, required=False):
        values = [
            plan_entry.pop(key, None),
            plan_entry.pop(key + '_' + env, None),
        ]
        self.pop_other_envs(env, plan_entry, key)

        values = [v for v in values if v is not None]
        if len(values) == 0:
            if required:
                raise Exception("Plan entry has no required %s parameter" % key)
            return default
        if len(values) == 1:
            return values[0]
        if joiner is None:
            return values[-1]
        return joiner.join(values)

    def iter_plan_templated(self, env, plan, plan_vars, include_disabled):
        for plan_id, plan_entry in plan.items():
            enabled = self.pop_plan_entry_parameter(env, plan_entry, 'enabled', default=True)
            if not (enabled or include_disabled):
                continue
            template_vars = TEMPLATABLE_PLAN_ITEM_REGEX.findall(plan_id)
            if not template_vars:
                yield plan_id, plan_entry, plan_vars
                continue
            template_vars_value_lists = list()
            for var in template_vars:
                value = plan_vars[var]
                if var == 'env':
                    value = [value]
                if not isinstance(value, list):
                    raise Exception("Template var %s is not list!" % var)
                template_vars_value_lists.append(value)
            for template_vars_values in itertools.product(*template_vars_value_lists):
                template_vars_value_instance_dict = dict(zip(template_vars, template_vars_values))
                plan_id_templated = jinja2.Template(plan_id, undefined=jinja2.StrictUndefined)\
                    .render(**template_vars_value_instance_dict)
                plan_vars_templated = plan_vars.copy()
                plan_vars_templated.update(template_vars_value_instance_dict)
                yield plan_id_templated, plan_entry.copy(), plan_vars_templated

    def get_plan_item(self, env, plan_id, plan_entry, plan_vars, strict):
        environment = environments.ENVIRONMENTS[env]
        command = self.pop_plan_entry_parameter(env, plan_entry, 'command', default='')
        args = self.pop_plan_entry_parameter(env, plan_entry, 'args', required=True, joiner=' ')
        args = args.replace('\n', ' ').strip()  # Sandbox steals spaces in the begining and end
        args = jinja2.Template(args, undefined=jinja2.StrictUndefined).render(**plan_vars)

        yt_trigger = self.pop_plan_entry_parameter(env, plan_entry, 'yt_trigger')
        if yt_trigger:
            if not isinstance(yt_trigger, list):
                yt_trigger = [yt_trigger]

            triggers = []
            for single_yt_trigger in yt_trigger:
                if isinstance(single_yt_trigger, str):
                    path = single_yt_trigger
                    min_age = None
                    max_age = None
                    creation_date_from = None
                    name_from = None
                    table_name_exclude_pattern = None
                    ignore_modifications = False
                else:
                    path = self.pop_plan_entry_parameter(env, single_yt_trigger, 'path', required=True)
                    min_age = self.pop_plan_entry_parameter(env, single_yt_trigger, 'min_age', required=False)
                    max_age = self.pop_plan_entry_parameter(env, single_yt_trigger, 'max_age', required=False)
                    creation_date_from = self.pop_plan_entry_parameter(env, single_yt_trigger, 'creation_date_from', required=False)
                    name_from = self.pop_plan_entry_parameter(env, single_yt_trigger, 'name_from', required=False)
                    table_name_exclude_pattern = self.pop_plan_entry_parameter(env, single_yt_trigger, 'table_name_exclude_pattern', required=False)
                    ignore_modifications = self.pop_plan_entry_parameter(env, single_yt_trigger, 'ignore_modifications', default=False, required=False)
                triggers.append(YtTriggerConfig(
                    path=jinja2.Template(path, undefined=jinja2.StrictUndefined).render(**plan_vars),
                    min_age=TimeInterval(min_age).to_timedelta() if min_age is not None else None,
                    max_age=TimeInterval(max_age).to_timedelta() if max_age is not None else None,
                    creation_date_from=datetime.datetime.fromisoformat(creation_date_from) if creation_date_from is not None else None,
                    name_from=name_from,
                    table_name_exclude_pattern=table_name_exclude_pattern,
                    ignore_modifications=ignore_modifications
                ))

            yt_trigger = triggers

        semaphore_name = self.pop_plan_entry_parameter(env, plan_entry, 'semaphore_name',
                                                       default=f'{plan_id}.{environment.released_at}')
        semaphore_name = jinja2.Template(semaphore_name, undefined=jinja2.StrictUndefined).render(**plan_vars)
        notify_email = self.pop_plan_entry_parameter(env, plan_entry, 'notify_email', default=None)
        if notify_email:
            notify_email = jinja2.Template(notify_email, undefined=jinja2.StrictUndefined).render(**plan_vars)
        else:
            notify_email = environment.notify_email

        tags = self.pop_plan_entry_parameter(env, plan_entry, 'tags', default=list())
        tags = list(environment.tags) + ['PLAN_ID:' + plan_id] + tags
        tags = tuple(sorted(tag.upper() for tag in tags))

        plan_item = PlanItem(
            at=self.pop_plan_entry_parameter(env, plan_entry, 'cron', default=None),
            yt_trigger=yt_trigger,
            plan_id=plan_id,
            resource=self.pop_plan_entry_parameter(env, plan_entry, 'resource', required=True),
            do_extract=self.pop_plan_entry_parameter(env, plan_entry, 'do_extract', default=False),
            requires_dns64=self.pop_plan_entry_parameter(env, plan_entry, 'requires_dns64', default=False),
            command=command,
            args=args,
            semaphore_name=semaphore_name,
            concurrency=self.pop_plan_entry_parameter(env, plan_entry, 'concurrency', default=1),
            kill_timeout=TimeInterval(self.pop_plan_entry_parameter(env, plan_entry, 'kill_timeout', default='3h')),
            task_type=self.pop_plan_entry_parameter(env, plan_entry, 'task_type', default=DEFAULT_TASK_TYPE),
            tags=tags,
            output_resources_file=self.pop_plan_entry_parameter(env, plan_entry, 'output_resources_file', default=''),
            notify_email=notify_email,
            notify_on_success=False,
            container_resource=self.pop_plan_entry_parameter(env, plan_entry, 'container_resource'),
        )

        at_is_set = bool(plan_item.at)
        yt_trigger_is_set = bool(plan_item.yt_trigger)

        if not at_is_set and not yt_trigger_is_set:
            raise Exception(f'{plan_id}: At least on of "cron", "yt_trigger" is required')

        if yt_trigger_is_set:
            for yt_trigger_config in plan_item.yt_trigger:
                proxy = yt_trigger_config.path.split('.')[0]
                if proxy not in self.yt_clients:
                    raise Exception(f'{plan_id}: unknown yt proxy "{proxy}"')

                if yt_trigger_config.path.endswith('/*'):
                    if at_is_set:
                        raise Exception(f'{plan_id}: "cron" is not compatible with "yt_trigger"/*')
                else:
                    for value, name in [(yt_trigger_config.min_age, 'min_age'),
                                        (yt_trigger_config.max_age, 'max_age'),
                                        (yt_trigger_config.creation_date_from, 'creation_date_from'),
                                        (yt_trigger_config.name_from, 'name_from'),
                                        (yt_trigger_config.table_name_exclude_pattern, 'table_name_exclude_pattern')]:
                        if value is not None:
                            raise Exception(f'{plan_id}: {name} can be used only with directory yt_trigger (ending with "/*")')

        self.pop_plan_entry_parameter(env, plan_entry, 'description')  # Unused for now TODO: Pass to task
        if strict and plan_entry:
            raise Exception("Non-recognized plan options: %s" % str(plan_entry.keys()))
        return plan_item

    def parse_plan(
        self,
        env: str,
        plan_data: RawPlan,
        strict: bool,
        include_disabled: bool = False,
        only_plan_item_id: Optional[str] = None,
    ) -> Dict[str, PlanItem]:
        plan = dict()
        known_ids = dict()
        for plan_file in plan_data.files:
            for plan_item_id, plan_item in yaml.safe_load(plan_file.data).items():
                if plan_item_id in known_ids:
                    raise Exception(f'Same plan item id declared in {plan_file.name} and {known_ids[plan_item_id]}')
                plan[plan_item_id] = plan_item
                known_ids[plan_item_id] = plan_file.name

        plan_vars = {
            'env': env,
        }
        plan_vars.update(plan.pop('_vars', {}))
        plan_vars.update(plan.pop('_vars_%s' % env, {}))
        self.pop_other_envs(env, plan, '_vars')

        result_plan = OrderedDict()
        plan_entries = self.iter_plan_templated(env, plan, plan_vars, include_disabled)
        for plan_id_templated, plan_entry, plan_vars_templated in plan_entries:
            if only_plan_item_id is not None and plan_id_templated != only_plan_item_id:
                continue
            try:
                plan_item = self.get_plan_item(env, plan_id_templated, plan_entry, plan_vars_templated, strict)
                result_plan[plan_id_templated] = plan_item
            except Exception as e:
                logging.error("Failed parse plan id '%s': %s" % (plan_id_templated, e))
                raise
        return result_plan

    def download_plan_from_yt(self) -> RawPlan:
        candidates = list()
        for px, yt_client in self.yt_clients.items():
            path = ytlib.join(self.environment.yt_root, 'plan')
            try:
                revision = yt_client.get(ytlib.join(path, self.YT_REVISION_ATTRIBUTE))
                candidates.append((revision, px, path))
            except Exception as e:
                logging.warning(f"Failed to get mod time at {px}:{path}, {e}")
        candidates.sort(reverse=True)
        for revision, px, path in candidates:
            logging.info(f"Trying to download plan from {px}:{path}, revision is {revision}")
            try:
                yt_client = self.yt_clients[px]
                plan_files = list()
                for file_path in yt_client.list(path, absolute=True):
                    name = ytlib.ypath_split(file_path)[1]
                    plan_data_binary = self.yt_clients[px].read_file(file_path).read()
                    data = plan_data_binary.decode('utf-8')
                    plan_files.append(PlanFile(name, data))

                logging.info(f"Successfully got plan from {px}:{path}")
                return RawPlan(plan_files, revision)
            except Exception as e:
                logging.warning(f"Failed to download plan from {px}:{path}, {e}")
                logging.warning(traceback.format_exc())
        raise Exception("Plan is not available at all sources")

    def upload_plan_to_yt(self, plan_data: RawPlan) -> None:
        glets = []
        for px, yt_client in self.yt_clients.items():
            glets.append(gevent.spawn(self.upload_files_to_yt, px, yt_client, plan_data))
        gevent.joinall(glets)
        if not any(map(lambda g: g.value, glets)):
            raise Exception("Failed to write plan to all YT clusters")

    def upload_files_to_yt(self, px: str, yt_client: ytlib.YtClient, plan_data: RawPlan):
        try:
            plan_files_path = ytlib.join(self.environment.yt_root, "plan")
            logging.info(f"Writing plan to {px}:{plan_files_path}...")
            with yt_client.Transaction():
                if not yt_client.exists(plan_files_path):
                    yt_client.mkdir(plan_files_path, recursive=True)
                for plan_file in plan_data.files:
                    path = ytlib.join(plan_files_path, plan_file.name)
                    yt_client.write_file(path, plan_file.data.encode('utf-8'))
                if plan_data.commit_revision:
                    logging.info(f"Writing commit revision to {px}:{plan_files_path}")
                    yt_client.set(ytlib.join(plan_files_path, self.YT_REVISION_ATTRIBUTE), plan_data.commit_revision)
            logging.info(f"Written plan to {px}:{plan_files_path}")
            return True
        except Exception as e:
            logging.warning("Failed to write plan at proxy %s: %s" % (px, e))
            return False

    def read_plan_from_files(self):
        path = Path(self.plan_path)
        if not path.exists():
            raise Exception(f"Failed to read plan. Path not exists {path}")
        if not path.is_dir():
            raise Exception(f"Failed to read plan. Path should be dir {path}")
        logging.info(f"Reading plan from {path}")
        plan_files = list()
        for file_path in path.glob("*.yaml"):
            with file_path.open('rt') as f:
                data = f.read()
            plan_files.append(PlanFile(file_path.stem, data))

        return RawPlan(plan_files, 0)

    def read_plan(self) -> RawPlan:
        if self.plan_path == 'yt':
            plan = self.download_plan_from_yt()
        else:
            plan = self.read_plan_from_files()
        if not plan.files:
            raise Exception("Failed to read plan. No files found")
        return plan

    def handle_render_plan(self):
        plan = self.parse_plan(self.env, self.read_plan(), strict=True)
        now = datetime.datetime.now(tz=self.TZ)
        for plan_id, plan_item in plan.items():
            next_run = '---'
            if plan_item.at:
                next_run = croniter(plan_item.at, now).get_next(datetime.datetime).isoformat()
            print(f'PlainId: {plan_id}, next run at {next_run}')
            for key, value in plan_item._asdict().items():
                if value is not None:
                    print("        %s: %s" % (key, value))

    def handle_update_plan_internal(self, notifier, notify_env, commit_revision, allow_downgrade):
        notifier.report_status(f'DeployingTo{notify_env}')
        try:
            if not allow_downgrade:
                old_plan = self.download_plan_from_yt()
                if commit_revision <= old_plan.commit_revision:
                    logging.info(f'Current plan revision in YT is {old_plan.commit_revision}, '
                                 f'will NOT overwrite it with older revision {commit_revision}')
                    notifier.report_status(f'DeployTo{notify_env}Cancelled')
                    return
            plan_data = self.read_plan()
            plan_data = plan_data._replace(commit_revision=commit_revision)
            self.validate_plan(plan_data, check_delegation=True)
            self.upload_plan_to_yt(plan_data)
            plan = self.parse_plan(self.env, plan_data, strict=True)
            # Нужно применить новый план, перенастроить шедулеры - прямо здесь и сейчас
            # Чтобы это корректно работало, надо чтобы update_plan_internal запускался под двумя семафорами:
            #   от продового и от тестингового планнера
            self.process_cron_triggers(plan, 0, 60)
            notifier.report_status(f'DeployedTo{notify_env}')
        except:
            notifier.report_status(f'DeployTo{notify_env}Failed')
            raise

    def handle_validate_plan(self, check_delegation):
        plan_data = self.read_plan()
        self.validate_plan(plan_data, check_delegation)

    def validate_plan(self, plan_data: RawPlan, check_delegation: bool) -> None:
        secrets = set()
        plan = self.parse_plan(self.env, plan_data, strict=True)
        for plan_id, plan_entry in plan.items():
            for arg in plan_entry.args.split(' '):
                if arg.startswith('sec-'):
                    secrets.add(Runner.parse_secret(arg)[0])
        if check_delegation:
            sb_tools.ensure_secrets_delegated(self.sb_client, secrets)

    @staticmethod
    def is_scheduler_compatible_with_plan_item(scheduler, plan_item):
        return sb_tools.extract_task_type_from_scheduler(scheduler) == plan_item.task_type

    def retrieve_plan_item_for_scheduler(self, scheduler, plan_items_to_create):
        for pos, plan_item_new in enumerate(plan_items_to_create):
            if self.is_scheduler_compatible_with_plan_item(scheduler, plan_item_new):
                plan_items_to_create.pop(pos)
                return plan_item_new
        return None

    def retrieve_scheduler_for_plan_item(self, schedulers_stopped, plan_item):
        for pos, scheduler in enumerate(schedulers_stopped):
            if self.is_scheduler_compatible_with_plan_item(scheduler, plan_item):
                schedulers_stopped.pop(pos)
                return scheduler
        return None

    @staticmethod
    def iter_yt_trigger_paths(yt_trigger: List[YtTriggerConfig]) -> List[YtTriggerPath]:
        for yt_trigger_config in yt_trigger:
            yt_proxy, yt_path = yt_trigger_config.path.split('.')
            if yt_path.endswith('/*'):
                as_dir = True
                yt_path = yt_path[:-2]
            else:
                as_dir = False
            yield YtTriggerPath(yt_proxy, yt_path, as_dir, yt_trigger_config.min_age, yt_trigger_config.max_age, yt_trigger_config.name_from, yt_trigger_config.creation_date_from,
                                yt_trigger_config.table_name_exclude_pattern, yt_trigger_config.ignore_modifications)

    def process_yt_triggers(self, plan):
        yt_trigger = YtTrigger(self.yt_clients, self.environment.yt_root)
        task_ids_to_start = list()
        for plan_id, plan_item in plan.items():
            if not plan_item.yt_trigger:
                continue

            yt_trigger_paths = Runner.iter_yt_trigger_paths(plan_item.yt_trigger)
            for node_info in yt_trigger.actualize_and_find_new_nodes(plan_id, yt_trigger_paths, self.YT_TRIGGER_LIMIT):
                logging.info(f'New node {node_info}')
                plan_item_exact = plan_item.replace(
                    args=plan_item.args.replace(YT_TRIGGER_PATH_PLACEHOLDER, node_info.node),
                )
                task = sb_tools.create_task(
                    sb_client=self.sb_client,
                    plan_item=plan_item_exact,
                    task_type=plan_item.task_type,
                    environment=self.environment,
                )
                task_ids_to_start.append(task['id'])

        if task_ids_to_start:
            logging.info(f'Have {len(task_ids_to_start)} tasks to start immediately')
            resp = self.sb_client.batch.tasks.start.update(task_ids_to_start)
            logging.info(resp[0]['message'])

        yt_trigger.flush()

    def process_cron_triggers(self, plan, plan_from, plan_to):
        now = datetime.datetime.now(tz=self.TZ)
        plan_from = now + datetime.timedelta(minutes=plan_from)
        plan_to = now + datetime.timedelta(minutes=plan_to)

        plan_items = list()
        for plan_id, plan_item in plan.items():
            if plan_item.at:
                iterator = croniter(plan_item.at, plan_from)
                while True:
                    at = iterator.get_next(datetime.datetime)
                    if at > plan_to:
                        break
                    plan_item_exact = plan_item.replace(at=at)
                    plan_items.append(plan_item_exact)

        logging.info("Have %s plan items in window (%s..%s)" % (len(plan_items), plan_from, plan_to))

        schedulers_data = self.sb_client.scheduler.read(
            limit=len(plan_items) * 2 + 50,
            tags=self.environment.tags,
            all_tags=True
        )

        schedulers_started_by_plan_item_hash = dict()  # plan_item hash -> scheduler
        schedulers_stopped = list()  # scheduler
        scheduler_ids_to_start = []
        scheduler_ids_to_remove = []

        scheduler_count_ok = 0

        # Получим список всех существующих шедулеров.
        # Запущенные разложим по параметрам запуска
        # Не запущенные тоже запомним, для переиспользования
        glets = []
        for scheduler_data in schedulers_data['items']:
            def read_scheduler(scheduler_data):
                if scheduler_data['status'] in ('WATCHING', 'WAITING') and scheduler_data['time']['next']:
                    at = dateutil.parser.isoparse(scheduler_data['schedule']['start_time'])
                    if plan_from <= at <= plan_to:
                        logging.info('Reading scheduler %s' % scheduler_data['id'])
                        scheduler = self.sb_client.scheduler[scheduler_data['id']].read()
                        plan_item_hash = sb_tools.extract_plan_item_hash_from_scheduler(scheduler)
                        if plan_item_hash is None:
                            logging.warning("Scheduler id %s has no plan item hash, remove it" % scheduler['id'])
                            scheduler_ids_to_remove.append(scheduler['id'])
                        else:
                            other_scheduler = schedulers_started_by_plan_item_hash.get(plan_item_hash)
                            if other_scheduler is not None:
                                logging.warning("Scheduler id %s has same params as %s, will remove last one" %
                                                (other_scheduler['id'], scheduler['id']))
                                scheduler_ids_to_remove.append(scheduler['id'])
                            else:
                                schedulers_started_by_plan_item_hash[plan_item_hash] = scheduler_data
                    else:
                        logging.info("Skipping scheduler %s for %s (out of plan-time)" % (scheduler_data['id'], at))
                else:
                    logging.info('Reading stopped scheduler %s' % scheduler_data['id'])
                    scheduler = self.sb_client.scheduler[scheduler_data['id']].read()
                    schedulers_stopped.append(scheduler)
            glets.append(gevent.spawn(read_scheduler, scheduler_data))
        gevent.joinall(glets, raise_error=True)
        logging.info("Read all schedulers")

        # Обойдем расписание, и вычислим, для каких записей нет готового шедулера,
        # а для каких есть
        plan_items_to_create = list()  # этим записям нужно сделать шедулер
        schedulers_to_create = list()
        schedulers_to_update = dict()
        for plan_item in plan_items:
            scheduler = schedulers_started_by_plan_item_hash.pop(plan_item.calc_hash(), None)
            if scheduler is None:
                plan_items_to_create.append(plan_item)
                logging.info(f"Cannot find scheduler for plan item {plan_item}")
            else:
                logging.info(f"Scheduler {scheduler['id']} is OK")
                logging.info(f"     plan : {plan_item}")
                scheduler_count_ok += 1

        # Обойдем оставшиеся шедулеры - часть переиспользуем, часть запланируем к остановке
        for plan_item_hash_old, scheduler in schedulers_started_by_plan_item_hash.items():
            plan_item_new = self.retrieve_plan_item_for_scheduler(scheduler, plan_items_to_create)
            if plan_item_new:
                logging.info(f"Will reuse active scheduler {scheduler['id']}")
                logging.info(f"     New  : {plan_item_new}")
                schedulers_to_update[scheduler['id']] = plan_item_new
            else:
                logging.info(f"Need to remove scheduler {scheduler['id']}")
                scheduler_ids_to_remove.append(scheduler['id'])

        # Вдруг остались записи плана без шедулера, создадим или переиспользуем для них шедулеры
        for plan_item in plan_items_to_create:
            scheduler = self.retrieve_scheduler_for_plan_item(schedulers_stopped, plan_item)
            if scheduler:
                logging.info(f"Will reuse stopped scheduler {scheduler['id']}")
                logging.info(f"     plan : {plan_item}")
                schedulers_to_update[scheduler['id']] = plan_item
                scheduler_ids_to_start.append(scheduler['id'])
            else:
                logging.info("Need to create scheduler for %s" % str(plan_item))
                schedulers_to_create.append(plan_item)

        scheduler_count_modified = len(schedulers_to_update)
        scheduler_count_created = len(schedulers_to_create)

        if schedulers_to_create:
            def create_scheduler(plan_item):
                scheduler_id = sb_tools.create_scheduler(self.sb_client, plan_item.task_type, self.environment)
                schedulers_to_update[scheduler_id] = plan_item
                scheduler_ids_to_start.append(scheduler_id)
            logging.info(f"Creating {len(schedulers_to_create)} schedulers")
            gevent.joinall(
                [gevent.spawn(create_scheduler, plan_item) for plan_item in schedulers_to_create],
                raise_error=True
            )

        if schedulers_to_update:
            logging.info(f"Updating {len(schedulers_to_update)} schedulers")
            gevent.joinall(
                [gevent.spawn(sb_tools.update_scheduler, self.sb_client, scheduler_id, plan_item, self.environment)
                            for scheduler_id, plan_item in schedulers_to_update.items()],
                raise_error=True
            )

        # Старые остановленные шедулеры нам не нужны!
        # Удалять сразу - тоже можно, но жалко, шедулер может скоро пригодиться. Экономим scheduler_id-ы, вобщем :-)
        for scheduler in schedulers_stopped:
            if (now - dateutil.parser.isoparse(scheduler['time']['updated'])).days > 0:
                logging.info(f"Will remove stopped scheduler {scheduler['id']}, because it is too old")
                scheduler_ids_to_remove.append(scheduler['id'])

        scheduler_count_deleted = len(scheduler_ids_to_remove)

        # поудаляем лишние шедулеры
        if scheduler_ids_to_remove:
            scheduler_ids_to_remove = sorted(scheduler_ids_to_remove)
            logging.info(f"Removing schedulers {scheduler_ids_to_remove}")
            self.sb_client.batch.schedulers.__getattr__('delete').update(scheduler_ids_to_remove)

        # позапускаем нужные шедулеры
        if scheduler_ids_to_start:
            logging.info(f"Starting schedulers {scheduler_ids_to_start}")
            self.sb_client.batch.schedulers.start.update(scheduler_ids_to_start)

        logging.info("Final schedulers stat: ")
        logging.info(f"  OK:       {scheduler_count_ok}")
        logging.info(f"  Created:  {scheduler_count_created}")
        logging.info(f"  Modified: {scheduler_count_modified}")
        logging.info(f"  Deleted:  {scheduler_count_deleted}")

    def handle_regular_run(self, plan_from, plan_to):
        plan_data = self.read_plan()
        plan = self.parse_plan(self.env, plan_data, strict=False)  # intentionally for fwd-compatibility
        self.process_yt_triggers(plan)
        self.process_cron_triggers(plan, plan_from, plan_to)

    def handle_drop_old_schedulers(self):
        while True:
            schedulers_data = self.sb_client.scheduler.read(
                limit=500,
                tags=self.environment.tags,
                all_tags=True,
                status=['STOPPED']
            )
            total = schedulers_data['total']
            items = schedulers_data['items']
            scheduler_ids_to_remove = [scheduler['id'] for scheduler in items]
            if len(scheduler_ids_to_remove) == 0:
                break
            logging.info(f"Removing {len(scheduler_ids_to_remove)} schedulers of {total}")
            self.sb_client.batch.schedulers.__getattr__('delete').update(scheduler_ids_to_remove)
            logging.info("Sleeping")
            time.sleep(10)

    def handle_prepare_dashboard(self, col_count, dashboard_path):
        plan_data = self.read_plan()
        plan_items_all = self.parse_plan('prod', plan_data, strict=True, include_disabled=True).keys()
        plan_items_by_env = {env: self.parse_plan(env, plan_data, strict=True).keys()
                             for env in environments.ENVIRONMENTS.keys()}

        grid = list()
        row = list()
        for plan_id in plan_items_all:
            if any(plan_id in items for items in plan_items_by_env.values()):
                row.append(plan_id)
                if len(row) == col_count:
                    grid.append(row)
                    row = list()
            else:
                logging.info(f'Skip plan item {plan_id}, because it is disabled in all environments')
        if row:
            grid.append(row)

        if not dashboard_path:
            dashboard_path = arcadia_tools.get_arcadia_path('travel', 'hotels', 'devops', 'solomon', 'dashboards', 'travel-sandbox.yaml')
        logging.info(f'Writing dashboard to {dashboard_path}')
        text = jinja2.Template(dedent(DASHBOARD_TEMPLATE)).render(grid=grid)
        with open(dashboard_path, 'w') as f:
            f.write(text)

    def handle_run_now(self, plan_id, yt_trigger_path, notify_email):
        plan_data = self.read_plan()
        plan = self.parse_plan(self.env, plan_data, strict=True, include_disabled=True, only_plan_item_id=plan_id)
        plan_item = plan.get(plan_id)
        if plan_item is None:
            raise Exception(f'No such plan_id "{plan_id}"')

        secrets = set()
        for arg in plan_item.args.split(' '):
            if arg.startswith('sec-'):
                secrets.add(Runner.parse_secret(arg)[0])
        sb_tools.ensure_secrets_delegated(self.sb_client, secrets)

        if yt_trigger_path:
            plan_item = plan_item.replace(args=plan_item.args.replace(YT_TRIGGER_PATH_PLACEHOLDER, yt_trigger_path))

        logging.info(f"Will send notifications to {notify_email}")

        plan_item = plan_item.replace(notify_email=notify_email, notify_on_success=True)
        self.plan_item_run_now(plan_item)

    def plan_item_run_now(self, plan_item):
        task = sb_tools.create_task(
            sb_client=self.sb_client,
            plan_item=plan_item,
            task_type=plan_item.task_type,
            environment=self.environment,
        )
        resp = self.sb_client.batch.tasks.start.update([task['id']])
        logging.info(resp[0]['message'])
