# -*- coding: utf-8 -*-

import logging

from sandbox.common.types import notification


def extract_plan_item_hash_from_scheduler(scheduler):
    task = scheduler['task']
    task_params = dict()
    for cf in task['custom_fields']:
        task_params[cf['name']] = cf['value']
    return task_params.get('plan_item_hash')


def extract_task_type_from_scheduler(scheduler):
    return scheduler['task']['type']


def create_scheduler(sb_client, task_type, environment):
    scheduler = sb_client.scheduler.create({
        'task_type': task_type,
        'owner': environment.sb_owner,
    })
    scheduler_id = scheduler['id']
    logging.info("Created scheduler id %s" % scheduler_id)
    return scheduler_id


def update_scheduler(sb_client, scheduler_id, plan_item, environment):
    sb_client.scheduler[scheduler_id] = {
        'schedule': {
            'sequential_run': False,
            'repetition': None,
            'start_time': plan_item.at.isoformat(),
            'fail_on_error': False,
        },
        'owner': environment.sb_owner,
        'task': get_task_fields(plan_item, environment)
    }
    logging.info("Updated scheduler %s" % scheduler_id)


def create_task(sb_client, plan_item, task_type, environment):
    logging.info(f'Creating task for {plan_item}')
    task_fields = get_task_fields(plan_item, environment)
    task = sb_client.task(type=task_type, **task_fields)
    task_id = task['id']
    logging.info(f'Created task https://sandbox.yandex-team.ru/task/{task_id}/view')
    return task


def get_task_fields(plan_item, environment):
    fields = {
        'tags': plan_item.tags,
        'kill_timeout': plan_item.kill_timeout.interval,
        'owner': environment.sb_owner,
        'custom_fields': [
            {
                'name': 'binary_resource_type',
                'value': plan_item.resource,
            },
            {
                'name': 'released_at',
                'value': environment.released_at
            },
            {
                'name': 'do_extract',
                'value': plan_item.do_extract,
            },
            {
                'name': 'requires_dns64',
                'value': plan_item.requires_dns64,
            },
            {
                'name': 'command',
                'value': plan_item.command,
            },
            {
                'name': 'args',
                'value': [plan_item.args],
            },
            {
                'name': 'semaphore_name',
                'value': plan_item.semaphore_name,
            },
            {
                'name': 'semaphore_capacity',
                'value': plan_item.concurrency,
            },
            {
                'name': 'enable_solomon',
                'value': True,
            },
            {
                'name': 'solomon_common_labels',
                'value': {
                    'plan_id': plan_item.plan_id,
                },
            },
            {
                'name': 'output_resources_file',
                'value': plan_item.output_resources_file,
            },
            {
                'name': 'notify_email',  # That is about custom notifications
                'value': plan_item.notify_email,
            },
            {
                'name': 'notify_caption',
                'value': "%s (SBPlanner)" % plan_item.plan_id,
            },
            {
                'name': 'notify_on_success',
                'value': plan_item.notify_on_success
            },
            {
                'name': 'plan_item_hash',
                'value': plan_item.calc_hash()
            },
        ],
        'notifications': [  # That is about built-in notifications
            {
                # This may duplicate notifications (custom + built-in), but twice is better than never
                'statuses': ['NO_RES', 'EXCEPTION', 'EXPIRED', 'STOPPED'],
                'transport': notification.Transport.EMAIL,
                'recipients': [plan_item.notify_email],
            },
        ],
    }
    if plan_item.container_resource is not None:
        fields['requirements'] = {
            'container_resource': plan_item.container_resource,
        }

    return fields


def ensure_secrets_delegated(sb_client, secrets):
    if not secrets:
        return
    res = sb_client.yav.status({'secrets': [{'id': sec} for sec in secrets]})
    to_delegate = list()
    for item in res['items']:
        if not item['delegated']:
            to_delegate.append(item['secret'])
    if to_delegate:
        logging.info("Not delegated secrets: %s, delegating" % to_delegate)
        res = sb_client.yav.tokens({'secrets': [{'id': sec} for sec in to_delegate]})
        for item in res['items']:
            if not item['delegated']:
                raise Exception("Failed to delegate secret: %s" % item)
    else:
        logging.info("All secrets are delegated")
