from travel.hotels.devops.slack_forwarder.duty_registry import DutyConfig

from typing import List, Optional
from datetime import datetime, timedelta

import requests


class AbcService:
    class DutyRole:
        def __init__(self, role_name: str, user_login: Optional[str], user_name: Optional[str]):
            self.role_name = role_name
            self.user_login = user_login
            self.user_name = user_name

    def __init__(self, token):
        self._token = token

    def get_duty_roles(self, duty_config: DutyConfig) -> List[DutyRole]:
        on_duty = self._get_on_duty(duty_config.service_id)
        shifts = []
        if duty_config.allow_future_shifts:
            date_from = self._format_date(datetime.now())
            date_to = self._format_date(datetime.now() + timedelta(7))
            shifts = self._get_shifts(duty_config.service_id, date_from, date_to)['results']

        seen_schedules = set()
        res = []

        for duty_info in on_duty + shifts:
            curr_schedule_id = duty_info['schedule']['id']
            if curr_schedule_id in seen_schedules:
                continue
            seen_schedules.add(curr_schedule_id)
            if duty_config.schedule_ids is not None and duty_info['schedule']['id'] not in duty_config.schedule_ids:
                continue
            role_name = duty_info['schedule']['name']
            if duty_info['schedule'].get('role') is not None:
                role_name = duty_info['schedule']['role']['name']['ru']
            person = duty_info.get('person')
            if person is not None:
                user_name = '{} {}'.format(person['first_name']['ru'], person['last_name']['ru'])
                res.append(AbcService.DutyRole(role_name, person['login'], user_name))
            else:
                res.append(AbcService.DutyRole(role_name, None, None))

        return res

    def _get_on_duty(self, service_id):
        url = 'https://abc-back.yandex-team.ru/api/v4/duty/on_duty/'
        params = {
            'service': service_id,
            'fields': 'schedule,person',
        }
        headers = {
            'Authorization': f'OAuth {self._token}'
        }

        return requests.get(url, params=params, headers=headers).json()

    def _get_shifts(self, service_id, date_from, date_to):
        url = 'https://abc-back.yandex-team.ru/api/v4/duty/shifts/'
        params = {
            'service': service_id,
            'date_from': date_from,
            'date_to': date_to,
            'fields': 'schedule,person',
        }
        headers = {
            'Authorization': f'OAuth {self._token}'
        }

        return requests.get(url, params=params, headers=headers).json()

    def _format_date(self, date):
        return datetime.strftime(date, '%Y-%m-%d')
