# Partially copied from https://a.yandex-team.ru/arc/trunk/arcadia/repo/pciexpress/arc_api/arc_api.py?blame=true&rev=7002164

from typing import Generator, List, Tuple

import grpc

import arc.api.public.repo_pb2
import arc.api.public.shared_pb2
import arc.api.public.repo_pb2_grpc


# DTO classes - protobuf counterparts needed for typehinting
class Timestamp:
    __slots__ = ('seconds', 'nanos')

    def __init__(self, seconds: int, nanos: int):
        self.seconds = seconds
        self.nanos = nanos


# noinspection PyPep8Naming
class Commit:
    __slots__ = ('Oid', 'TreeOid', 'ParentOids', 'Timestamp', 'Author', 'Message', 'SvnRevision')

    def __init__(self, Oid: str, TreeOid: str, ParentOids: List[str], Timestamp: Timestamp, Author: str, Message: str,
                 SvnRevision: int):
        self.Oid = Oid
        self.TreeOid = TreeOid
        self.ParentOids = ParentOids
        self.Timestamp = Timestamp
        self.Author = Author
        self.Message = Message
        self.SvnRevision = SvnRevision


# noinspection PyPep8Naming
class Ref:
    __slots__ = ('Name', 'Commit', 'LastChanged')

    def __init__(self, Name: str, Commit: Commit, LastChanged: Timestamp):
        self.Name = Name
        self.Commit = Commit
        self.LastChanged = LastChanged


# noinspection PyPep8Naming
class ReadFileHeader:
    __slots__ = ('Oid', 'FileSize')

    def __init__(self, Oid: str, FileSize: int):
        self.Oid = Oid
        self.FileSize = FileSize


class ArcApi:
    def __init__(self, token: str, url: str = 'api.arc-vcs.yandex-team.ru:6734'):
        self._channel = self._connect(url, token)

        self._branch_service = arc.api.public.repo_pb2_grpc.BranchServiceStub(self._channel)
        self._file_service = arc.api.public.repo_pb2_grpc.FileServiceStub(self._channel)

        self.timeout = 120  # seconds

    @staticmethod
    def _connect(url, token):
        token_credentials = grpc.access_token_call_credentials(token)
        channel_credentials = grpc.composite_channel_credentials(grpc.ssl_channel_credentials(), token_credentials)
        return grpc.secure_channel(url, channel_credentials)

    def read_file(self, revision, path) -> Tuple[ReadFileHeader, bytes]:
        request = arc.api.public.repo_pb2.ReadFileRequest(Revision=revision, Path=path)
        response = self._file_service.ReadFile(request, timeout=self.timeout)

        file_content = b''
        header = None

        for batch in response:
            if batch.HasField('Header'):
                header = batch.Header
            else:
                file_content += batch.Data

        return header, file_content

    def list_refs(self, prefix_filter) -> Generator[Ref, None, None]:
        request = arc.api.public.repo_pb2.ListRefsRequest(PrefixFilter=prefix_filter, Inconsistent=True)
        for batch in self._branch_service.ListRefs(request, timeout=self.timeout):
            for ref in batch.Refs:
                yield Ref(ref.Name, ref.Commit, ref.LastChanged)
