# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import time
from dataclasses import dataclass
from datetime import datetime, timedelta

import pytz
from jinja2 import Template
from ids.registry import registry

from travel.hotels.devops.slack_forwarder.app import app
from travel.hotels.devops.slack_forwarder.notifier_worker import send_plaintext_notification

TRAVEL_DEPARTMENT_URL = 'yandex_infra_data_3819'
BIRTHDAYS_CHANNEL = 'travel-birthdays'
MSK_TZ = pytz.timezone('Europe/Moscow')
MESSAGE_TEMPLATE = Template('''
Привет!

Сегодня дни рождения у:
  {% for human in result.today %}
    * {{ human }}
  {% endfor %}

Завтра дни рождения у:
  {% for human in result.tomorrow %}
    * {{ human }}
  {% endfor %}
''')


@dataclass
class Result:
    today: list[str]
    tomorrow: list[str]

    def has_birthdays(self):
        return bool(self.today or self.tomorrow)


class BirthdaysReporter:
    def __init__(self, staff_token):
        self._staff_token = staff_token
        self._last_sent = None

    def run(self):
        with app.app_context():
            while True:
                try:
                    date_now = datetime.utcnow().date()
                    if self._last_sent is None or date_now > self._last_sent:
                        result = get_closest_birthdays(self._staff_token)
                        if result.has_birthdays():
                            send_plaintext_notification(MESSAGE_TEMPLATE.render(result=result), BIRTHDAYS_CHANNEL)
                        self._last_sent = date_now
                except Exception:
                    app.logger.error('Exception while checking birthdays', exc_info=True)
                time.sleep(3600)


def _get_children_departments(repository, department_url):
    departments = repository.get(lookup={
        'type': 'department',
        'ancestors.url': department_url,
    })
    result = [d['id'] for d in departments]
    for d in departments:
        result.extend(_get_children_departments(repository, d['url']))
    return result


def _get_department_ids(repository, root_department_url):
    root_department = repository.get(lookup={
        'type': 'department',
        'url': root_department_url,
    })
    result = [root_department[0]['id']]
    return result + _get_children_departments(repository, root_department_url)


def _get_member_logins(repository, department_id):
    return [m['person']['login'] for m in repository.get({
        'group.id': department_id,
        'person.official.is_dismissed': False,
    })]


def _get_birthdays(repository, logins):
    result = {}
    for login in logins:
        person = repository.get_one(lookup={'login': login})
        if person:
            result[login] = person['personal']['birthday']
    return result


def _find_closest_birthdays(birthdays_by_logins):
    result = Result(today=[], tomorrow=[])
    dt_now = pytz.UTC.localize(datetime.utcnow())
    msc_now_date = dt_now.astimezone(MSK_TZ).date()
    for login, birthday in birthdays_by_logins.items():
        birthday_dt = datetime.strptime(birthday, '%Y-%M-%d').date()
        if birthday_dt == msc_now_date:
            result.today.append(login)
        if birthday_dt == msc_now_date + timedelta(1):
            result.tomorrow.append(login)
    return result


def get_closest_birthdays(staff_token):
    persons = registry.get_repository('staff', 'person', user_agent='myservice', oauth_token=staff_token)
    departments = registry.get_repository('staff', 'group', user_agent='myservice', oauth_token=staff_token)
    memberships = registry.get_repository('staff', 'groupmembership', user_agent='myservice', oauth_token=staff_token)

    department_ids = _get_department_ids(departments, TRAVEL_DEPARTMENT_URL)
    logins_set = set()
    for department_id in department_ids:
        logins_set.update(set(_get_member_logins(memberships, department_id)))

    birthdays_by_logins = _get_birthdays(persons, logins_set)
    return _find_closest_birthdays(birthdays_by_logins)
