from functools import wraps

from flask import request, g, redirect
from werkzeug.exceptions import InternalServerError

from travel.hotels.devops.slack_forwarder.app import app

_MAX_AGE=2 * 60 * 60


class NeedRedirectError(Exception):
    def __init__(self, response):
        self.response = response


def blackbox(bb_client_getter):
    def _blackbox(f):
        @wraps(f)
        def _blackbox_middleware(*args, **kwargs):
            bb_response = _check_session_id(bb_client_getter())
            is_valid = _check_valid_response(bb_response)

            if not is_valid:
                error = bb_response['error']
                app.logger.error(f'Blackbox error: {error}')
                raise InternalServerError('Blackbox request error')

            need_redirect = _check_redirect(bb_response)

            if need_redirect:
                response = redirect(f'https://passport.yandex-team.ru/auth?retpath={request.url}')
                raise NeedRedirectError(response)

            # https://docs.yandex-team.ru/blackbox/methods/sessionid#response_format
            g.user = bb_response

            return f(*args, **kwargs)

        return _blackbox_middleware
    return _blackbox


def _check_session_id(bb_client):
    session_id = request.cookies.get('Session_id')
    user_ip = request.access_route[0] if request.access_route else '127.0.0.1'
    host = request.host
    if ':' in host:
        host = host.split(':')[0]
    return bb_client.sessionid(user_ip, session_id, host, get_user_ticket='yes')


def _check_valid_response(bb_response):
    status = bb_response['status']['value']
    return status == 'VALID' or status == 'NEED_RESET'


def _check_redirect(bb_response):
    status = bb_response['status']['value']
    age = bb_response.get('age', 0)
    return status == 'NEED_RESET' or (status == 'NOAUTH' and age < _MAX_AGE)
