import time
import datetime
import yaml
import uuid
import urllib.parse
from collections import defaultdict

from travel.hotels.devops.slack_forwarder import utils
from travel.hotels.devops.slack_forwarder.pg_db import pg_query_db
from travel.hotels.devops.slack_forwarder.arc_api import ArcApi
from travel.hotels.devops.slack_forwarder.app import app
from travel.hotels.devops.slack_forwarder.notifier import States, get_pretty_component_name
from travel.hotels.devops.slack_forwarder.nanny import NannyPauseStateManager


class CompositeComponentInfo:
    def __init__(self, component_name, raw_component_name):
        self.component_name = component_name
        self.raw_component_name = raw_component_name
        self.unique_id = str(uuid.uuid4())
        self.status_text = None
        self.revision_link_text = None
        self.sb_link_text = None
        self.nanny_link_text = None
        self.sb_search_link_text = None
        self.last_finished_task_id = None
        self.last_finished_revision = None
        self.commits = []
        self.pausable = False
        self.paused = False
        self.nanny_service_id = None


class CommitInfo:
    def __init__(self, revision, timestamp, message, author, task_id):
        self.revision = revision
        self.timestamp = timestamp
        self.message = message
        self.author = author
        self.task_id = task_id


class ComponentInfoBuilder:
    def __init__(self, arc_token):
        self._user = 'robot-travel-prod'
        self._arc_api = ArcApi(token=arc_token)
        self._components_info = None
        self._all_refs = None
        self._release_commits_by_component = None
        self._build_commits_by_component = None
        self._all_events = None
        self._all_pause_infos = None
        self._max_components_info_file_age_sec = 10 * 60  # 10 min
        self._max_refs_age_sec = 1 * 60  # 1 min
        self._max_events_age_sec = 1  # 1 sec
        self._max_pause_states_age_sec = 1  # 1 sec

    def run(self):
        last_components_info_file_check = 0
        last_refs_check = 0
        last_events_check = 0
        last_pause_check = 0
        with app.app_context():
            while True:
                try:
                    now = time.time()
                    if now - last_components_info_file_check > self._max_components_info_file_age_sec:
                        self._load_components_info()
                        last_components_info_file_check = now
                    if now - last_refs_check > self._max_refs_age_sec:
                        self._load_refs()
                        last_refs_check = now
                    if now - last_events_check > self._max_events_age_sec:
                        self._load_events()
                        last_events_check = now
                    if now - last_pause_check > self._max_pause_states_age_sec:
                        self._load_pause_data()
                        last_pause_check = now
                except Exception:
                    app.logger.error('Exception while updating info', exc_info=True)
                time.sleep(1)

    def _load_components_info(self):
        header, data = self._arc_api.read_file('trunk', 'testenv/jobs/travel/TravelBuild.yaml')
        self._components_info = yaml.full_load(data)

    def _load_refs(self):
        self._all_refs = list(self._arc_api.list_refs(f'users/{self._user}'))
        release_commits_by_component = defaultdict(list)
        build_commits_by_component = defaultdict(list)
        for ref in self._all_refs:
            if ref.Name.startswith(f'users/{self._user}/releases/stable/'):
                normalized_component_name = ref.Name.split('/')[4]
                release_commits_by_component[normalized_component_name].append(ref.Commit)
            if ref.Name.startswith(f'users/{self._user}/builds/'):
                normalized_component_name = ref.Name.split('/')[3]
                build_commits_by_component[normalized_component_name].append(ref.Commit)

        self._release_commits_by_component = release_commits_by_component
        self._build_commits_by_component = build_commits_by_component

    def _load_events(self):
        res = []
        events = pg_query_db('SELECT revision, component, event_type, task_id FROM events')
        for revision, component, event_type, task_id in events:
            try:
                revision = int(revision)
            except ValueError:
                continue
            res.append((revision, component, event_type, task_id))
        self._all_events = res

    def _load_pause_data(self):
        self._all_pause_infos = {service_id: status for service_id, status in NannyPauseStateManager().get_all_statuses()}

    def _get_info_by_components(self, is_allowed_state, is_allowed_last_state):
        revisions_blacklist = set()
        revision_status = dict()
        for revision, component, event_type, task_id in self._all_events:
            key = (component, revision)
            state = getattr(States, event_type)
            if not is_allowed_state(state):
                revisions_blacklist.add(key)
            if key not in revision_status or revision_status[key][0].priority < state.priority:
                revision_status[key] = (state, task_id)

        for key, (state, task_id) in revision_status.items():
            if not is_allowed_last_state(state):
                revisions_blacklist.add(key)

        for key in revisions_blacklist:
            if key in revision_status:
                revision_status.pop(key)

        result = {}
        for (component, revision), (state, task_id) in revision_status.items():
            if component not in result or result[component][2] < revision:
                result[component] = (state, task_id, revision)

        return result

    def _get_pause_status_by_components(self):
        return self._all_pause_infos

    def get_components_info(self):
        info_by_component = self._get_info_by_components(lambda state: state not in [States.BuildFiltered],
                                                         lambda state: True)

        info_by_component_ready_for_release = self._get_info_by_components(
            lambda state: state not in [States.BuildFiltered], lambda state: state.ready_for_release)

        pause_statuses = self._get_pause_status_by_components()

        component_name_to_nanny_service = {k: v[0] for k, v in app.config['NANNY_COMPONENTS'].items() if len(v) > 0}

        composite_infos = []
        for section_name, section in self._components_info.items():
            for name, item in section.get('multiple_jobs', dict()).items():
                component_name = item['component_name']
                resource_type = item['resource_type'] if (section_name == 'TRAVEL_PACKAGE' or section_name == 'TRAVEL_PACKAGE_2') else item['result_rt']
                composite_info = CompositeComponentInfo(get_pretty_component_name(component_name, unicode_emoji=True), component_name)
                if component_name in info_by_component:
                    state, task_id, revision = info_by_component[component_name]
                    revision_link = f'https://a.yandex-team.ru/arc/commit/{revision}'
                    sb_link = state.get_sb_link(task_id)
                    nanny_link = state.get_nanny_link(task_id)

                    composite_info.status_text = f'{state.pretty_name} {state.unicode_emoji}'
                    composite_info.revision_link_text = f'<a href="{revision_link}">r{revision}</a>'
                    composite_info.sb_link_text = f'<a href="{sb_link}">{task_id}</a>'
                    if nanny_link:
                        composite_info.nanny_link_text = f'<a href="{nanny_link}">nanny</a>'

                if component_name in info_by_component_ready_for_release:
                    commits = self._get_pending_component_commits_via_arc(component_name)
                    revisions_via_arc = [x.SvnRevision for x in commits]
                    state, task_id, revision = info_by_component_ready_for_release[component_name]
                    if not state.released_to_stable and revision in revisions_via_arc:
                        composite_info.last_finished_task_id = task_id
                        composite_info.last_finished_revision = revision
                        composite_info.commits = [x for x in self.get_component_commits(component_name) if x.revision <= revision]

                format = '%Y-%m-%dT%H:%M:%SZ'
                year_ago = datetime.datetime.utcnow() - datetime.timedelta(365)
                query = urllib.parse.urlencode({'type': resource_type, 'limit': 10, 'created': f'{year_ago.strftime(format)}..'})
                sb_search_link = f'https://sandbox.yandex-team.ru/resources?{query}'
                composite_info.sb_search_link_text = f'<a href="{sb_search_link}">sb search</a>'
                composite_info.pausable = component_name in component_name_to_nanny_service
                composite_info.paused = pause_statuses.get(component_name_to_nanny_service.get(component_name, ''), False)
                composite_info.nanny_service_id = component_name_to_nanny_service.get(component_name)
                composite_infos.append(composite_info)

        return composite_infos

    def get_component_commits(self, component_name):
        task_id_by_revision = {revision: task_id
                               for revision, component, _, task_id in self._all_events
                               if component == component_name}

        commits = self._get_pending_component_commits_via_arc(component_name)
        commits_table = []
        for commit in commits:
            commits_table.append(CommitInfo(commit.SvnRevision, commit.Timestamp, commit.Message, commit.Author,
                                            task_id_by_revision.get(commit.SvnRevision)))

        return commits_table

    def _get_pending_component_commits_via_arc(self, component_name):
        normalized_component_name = utils.normalize_component_name(component_name)
        release_commits = self._release_commits_by_component[normalized_component_name]
        build_commits = self._build_commits_by_component[normalized_component_name]

        last_release = sorted(release_commits, key=lambda x: x.SvnRevision)[-1] if len(release_commits) > 0 else None
        commits = [x for x in build_commits if last_release is None or x.SvnRevision > last_release.SvnRevision]
        commits = sorted(commits, key=lambda x: x.SvnRevision, reverse=True)
        return commits

    def get_pretty_component_name(self, component_name, emoji=True):
        return get_pretty_component_name(component_name, emoji=emoji, unicode_emoji=True)
