import time
from collections import defaultdict

from travel.hotels.devops.slack_forwarder.arc_api import ArcApi
from travel.hotels.devops.slack_forwarder.app import app
from travel.hotels.devops.slack_forwarder.utils import should_run_regular_check
from travel.hotels.devops.slack_forwarder.notifier import SlackLostCommitsReporter
from travel.hotels.devops.slack_forwarder.lost_commits_data import CommitInfo, LostCommitsInfo, LostCommitsInfoByAuthor, LostCommitsReport


class LostCommitsWatcher:
    def __init__(self, token):
        self._arc_api = ArcApi(token=token)
        self._slack_lost_commits_reporter = SlackLostCommitsReporter()
        self._last_check_time = 0
        self.ignored_components = app.config['COMPONENTS_WITHOUT_LOST_COMMITS_REPORT']

    def run(self):
        with app.app_context():
            while True:
                try:
                    self._check_lost_commits()
                except Exception:
                    app.logger.error('Exception while watching lost commits', exc_info=True)
                time.sleep(10)

    def _check_lost_commits(self, force=False):
        # Run check only in interval 11-12 MSK (13-14 EKB) on weekdays, and not not more often than each 2 hours
        if not force and not should_run_regular_check(self._last_check_time, from_utc_hour=8, to_utc_hour=9):
            return

        user = 'robot-travel-prod'

        refs = list(self._arc_api.list_refs(f'users/{user}'))

        app.logger.info(f'Got {len(refs)} refs from arc api')

        releases = defaultdict(list)
        for ref in refs:
            if ref.Name.startswith(f'users/{user}/releases/stable'):
                component = ref.Name.split('/')[4]
                releases[component].append(ref.Commit)

        components = defaultdict(list)
        for ref in refs:
            if ref.Name.startswith(f'users/{user}/builds'):
                component = ref.Name.split('/')[3]
                components[component].append(ref.Commit)

        app.logger.info(f'Extracted info about {len(components)} components from refs')

        by_components = []
        commits_by_authors = defaultdict(lambda: defaultdict(list))
        now_ts = int(time.time())
        delay = 5 * 24 * 60 * 60  # 5 days
        for component, commits in components.items():
            last_release = sorted(releases[component], key=lambda x: x.SvnRevision)[-1] if component in releases else None
            commits = [x for x in commits if last_release is None or x.SvnRevision > last_release.SvnRevision]
            old_commits = [x for x in commits if x.Timestamp.seconds < now_ts - delay]
            if len(old_commits) == 0:
                continue
            if component in self.ignored_components:
                continue
            commits = sorted(commits, key=lambda x: x.SvnRevision, reverse=True)
            by_components.append(LostCommitsInfo(component, last_release.Timestamp if last_release is not None else None,
                                 [CommitInfo(x.SvnRevision, x.Timestamp.seconds, x.Author) for x in commits]))

            for commit in old_commits:
                commits_by_authors[commit.Author][component].append(CommitInfo(commit.SvnRevision, commit.Timestamp.seconds, commit.Author))

        by_authors = dict()
        for author, components in commits_by_authors.items():
            by_authors[author] = []
            for component, commits in components.items():
                by_authors[author].append(LostCommitsInfoByAuthor(component, commits))

        report = LostCommitsReport(by_components, by_authors)

        app.logger.info('Built lost commits report')

        if len(report.by_components) > 0:
            self._slack_lost_commits_reporter.send_lost_commits_report(report)
            app.logger.info('Sent lost commits report')
        else:
            app.logger.info('Lost commits report is skipped due to no lost commits')

        self._last_check_time = time.time()
