import time
import datetime
import requests

from nanny_rpc_client import RequestsRpcClient
from nanny_tickets import tickets_api_stub
from nanny_tickets import tickets_api_pb2
from nanny_tickets import tickets_pb2

from travel.hotels.devops.slack_forwarder.app import app
from travel.hotels.devops.slack_forwarder.utils import get_days_form
from travel.hotels.devops.slack_forwarder.utils import should_run_regular_check
from travel.hotels.devops.slack_forwarder.pg_db import pg_query_db, pg_query_db_and_commit
from travel.hotels.devops.slack_forwarder.notifier import States
from travel.hotels.devops.slack_forwarder.notifier_worker import trigger_notification, send_plaintext_notification, send_nanny_update_report, send_pause_report


class NannyClient:
    def __init__(self, *, token=None, session_id=None):
        self._nanny_services_base_url = 'https://nanny.yandex-team.ru/v2/services'
        self._session = requests.Session()
        if token is not None:
            self._session.headers['Authorization'] = f'OAuth {token}'
        elif session_id is not None:
            self._session.cookies.set('Session_id', session_id)
        else:
            raise Exception('No token and no session_id provided')

    def get_services(self):
        return self._session.get(f'{self._nanny_services_base_url}/?exclude_runtime_attrs=1&category=/travel').json()['result']

    def get_runtime_attrs(self, service_id):
        return self._session.get(f'{self._nanny_services_base_url}/{service_id}/history/runtime_attrs/').json()

    def get_current_state(self, service_id):
        return self._session.get(f'{self._nanny_services_base_url}/{service_id}/current_state/').json()

    def pause_service(self, service_id, comment):
        app.logger.info(f'Pausing {service_id}')
        res = self._session.post(f'{self._nanny_services_base_url}/{service_id}/events/', json={"type": "PAUSE_ACTIONS", "content": {"comment": comment}})
        if not res.ok:
            raise Exception(f'Failed to pause service: {res.text}')

    def resume_service(self, service_id, comment):
        app.logger.info(f'Resuming {service_id}')
        res = self._session.post(f'{self._nanny_services_base_url}/{service_id}/events/', json={"type": "RESUME_ACTIONS", "content": {"comment": comment}})
        if not res.ok:
            raise Exception(f'Failed to resume service: {res.text}')

    def check_pause_state(self, service_id):
        pause_state_manager = NannyPauseStateManager()
        current_state = self.get_current_state(service_id)
        is_paused = current_state['content']['is_paused']['value']
        was_paused = pause_state_manager.is_paused_state(service_id)

        if was_paused != is_paused:
            if was_paused is not None:
                send_pause_report(service_id, is_paused, current_state['content']['is_paused']['info']['comment'], current_state['content']['is_paused']['info']['author'])
            pause_state_manager.update_state(service_id, is_paused)


class NannyPauseStateManager:
    def __init__(self):
        self._paused_state = 'paused'
        self._normal_state = 'normal'

    def is_paused_state(self, service_id):
        res = pg_query_db('SELECT status FROM nanny_service_statuses WHERE service_id = %s', [service_id])
        if len(res) == 0:
            return None
        return res[0][0] == self._paused_state

    def update_state(self, service_id, is_paused):
        args = [service_id, self._paused_state if is_paused else self._normal_state]
        pg_query_db_and_commit('INSERT INTO nanny_service_statuses (service_id, status) VALUES (%s, %s) ON CONFLICT (service_id) DO UPDATE SET service_id = %s, status = %s', args + args)

    def get_all_statuses(self):
        res = []
        statuses = pg_query_db('SELECT service_id, status FROM nanny_service_statuses')
        for service_id, status in statuses:
            res.append((service_id, status == self._paused_state))
        return res


class NannyWatcher:
    def __init__(self, token):
        self.client = RequestsRpcClient('http://nanny.yandex-team.ru/api/tickets', request_timeout=10, oauth_token=token)
        self._token = token
        self._last_resources_check_time = 0

        self._nanny_services_base_url = 'https://nanny.yandex-team.ru/v2/services'

    def run(self):
        with app.app_context():
            while True:
                try:
                    self._actualize_deployments_states()
                    self._check_override_resources()
                    self._check_nanny_updates()
                    self._check_nanny_service_statuses()
                except Exception:
                    app.logger.error('Exception while watching nanny', exc_info=True)
                time.sleep(10)

    def _check_override_resources(self, force=False):
        # Run check only in interval 10-11 MSK (12-13 EKB) on weekdays, and not not more often than each 2 hours
        if not force and not should_run_regular_check(self._last_resources_check_time, from_utc_hour=7, to_utc_hour=8):
            return

        nanny_client = NannyClient(token=self._token)
        services = nanny_client.get_services()

        for service in [x['_id'] for x in services]:
            if service in app.config['NO_OVERRIDE_CHECK_SERVICES']:
                continue
            runtime_attrs = nanny_client.get_runtime_attrs(service)
            curr_static_files = runtime_attrs['result'][0]['content']['resources']['static_files']
            paths = [file['local_path'] for file in curr_static_files if self._should_check_file(file)]
            for path in paths:
                last_not_valid_timestamp = None
                found_valid = False
                for state in runtime_attrs['result']:
                    files = [x for x in state['content']['resources']['static_files'] if x['local_path'] == path]
                    if len(files) == 0 or len(files[0]['content']) == 0:
                        found_valid = True
                        break
                    last_not_valid_timestamp = state['change_info']['ctime'] / 1000

                if found_valid:
                    not_empty_days = int((time.time() - last_not_valid_timestamp) / 60 / 60 / 24)
                    if not_empty_days >= 5:
                        send_plaintext_notification(f'Файл `{path}` в cервисе `{service}` не пуст уже {not_empty_days} {get_days_form(not_empty_days)}', 'devops')
                else:
                    last_state = runtime_attrs['result'][-1]
                    not_empty_days_at_least = int((time.time() - last_state['change_info']['ctime'] / 1000) / 60 / 60 / 24)
                    send_plaintext_notification(f'Файл `{path}` в cервисе `{service}` не пуст уже больше чем {not_empty_days_at_least} {get_days_form(not_empty_days_at_least)}', 'devops')

        self._last_resources_check_time = time.time()

    def _should_check_file(self, file):
        return self._is_override_file(file['local_path']) and len(file['content']) > 0

    def _is_override_file(self, path):
        return any(x in path for x in ['override', 'properties', 'application', 'config'])

    def _actualize_deployments_states(self):
        events = pg_query_db("SELECT component, event_type, task_id, revision, timestamp FROM events", [])
        states = {}
        for component, event_type, task_id, revision, timestamp in events:
            key = (component, revision)
            if key not in states or states[key]['state'].priority < getattr(States, event_type).priority:
                states[key] = {'state': getattr(States, event_type), 'task_id': task_id, 'timestamp': timestamp}

        state_mapping = {
            States.ReleasedToTesting: 'TESTING',
            States.DeployingToTesting: 'TESTING',
            States.ReleasedToStable: 'STABLE',
            States.DeployingToStable: 'STABLE',
            States.ReleasedToUnstable: 'UNSTABLE',
            States.DeployingToUnstable: 'UNSTABLE'
        }

        initial_states = [States.ReleasedToTesting, States.ReleasedToStable, States.ReleasedToUnstable]

        for ((component, revision), state) in states.items():
            if state['state'] not in state_mapping:
                continue
            timestamp = state['timestamp']
            if state['state'] in initial_states and timestamp < datetime.datetime.utcnow() - datetime.timedelta(0, 3600):
                continue
            self._check_deployment(component, revision, state_mapping.get(state['state']), state['task_id'])

    def _check_deployment(self, component, revision, env, task_id):
        tickets_stub = tickets_api_stub.TicketServiceStub(self.client)

        req = tickets_api_pb2.FindTicketsRequest()
        req.limit = 1
        req.query.release_id = f'SANDBOX_RELEASE-{task_id}-{env}'
        resp = tickets_stub.find_tickets(req)

        if resp.total == 0:
            return

        release = list(resp.value)[0].status
        status = release.status
        status_name = tickets_pb2.TicketStatus.Status.Name(status)
        start_time = getattr(getattr(release, 'start_time', None), 'seconds', None)
        end_time = getattr(getattr(release, 'end_time', None), 'seconds', None)
        current_state = pg_query_db('SELECT status, start_time, end_time FROM deployments WHERE task_id=%s AND env=%s', [task_id, env])
        args = [task_id, env, status_name, start_time, end_time]
        pg_query_db_and_commit('INSERT INTO deployments (task_id, env, status, start_time, end_time) VALUES (%s, %s, %s, %s, %s) ' +
                               'ON CONFLICT (task_id, env) DO UPDATE SET task_id = %s, env = %s, status = %s, start_time = %s, end_time = %s', args + args)
        if len(current_state) == 0 or current_state[0] != (status_name, start_time, end_time):
            if len(current_state) == 0 or status in [tickets_pb2.TicketStatus.CANCELLED, tickets_pb2.TicketStatus.DEPLOY_SUCCESS, tickets_pb2.TicketStatus.DEPLOY_FAILED]:
                self._mark_deploy_status(component, revision, env, task_id, status)
            trigger_notification(revision)

    def _mark_deploy_status(self, component, revision, env, task_id, status):
        state_mapping = {
            'TESTING': {
                tickets_pb2.TicketStatus.DEPLOY_SUCCESS: States.DeployedToTesting,
                tickets_pb2.TicketStatus.DEPLOY_FAILED: States.DeployToTestingFailed,
                tickets_pb2.TicketStatus.CANCELLED: States.DeployToTestingCancelled,
            },
            'STABLE': {
                tickets_pb2.TicketStatus.DEPLOY_SUCCESS: States.DeployedToStable,
                tickets_pb2.TicketStatus.DEPLOY_FAILED: States.DeployToStableFailed,
                tickets_pb2.TicketStatus.CANCELLED: States.DeployToStableCancelled,
            },
            'UNSTABLE': {
                tickets_pb2.TicketStatus.DEPLOY_SUCCESS: States.DeployedToUnstable,
                tickets_pb2.TicketStatus.DEPLOY_FAILED: States.DeployToUnstableFailed,
                tickets_pb2.TicketStatus.CANCELLED: States.DeployToUnstableCancelled,
            }
        }
        defaults_mapping = {
            'TESTING': States.DeployingToTesting,
            'STABLE': States.DeployingToStable,
            'UNSTABLE': States.DeployingToUnstable
        }
        if env not in state_mapping:
            raise Exception(f'Unknown env: {env}')
        pg_query_db_and_commit('INSERT INTO events (revision, component, event_type, task_id) VALUES (%s, %s, %s, %s)',
                               [revision, component, state_mapping[env].get(status, defaults_mapping[env]).name, task_id])

    def _check_nanny_updates(self):
        nanny_client = NannyClient(token=self._token)
        services = nanny_client.get_services()

        for service_id in [x['_id'] for x in services]:
            if 'prod' not in service_id:
                continue

            runtime_attrs = nanny_client.get_runtime_attrs(service_id)['result']
            nanny_releases = pg_query_db('SELECT last_snapshot_id FROM nanny_releases WHERE service_id = %s', [service_id])
            current_snapshot_id = runtime_attrs[0]['_id']
            releaser = runtime_attrs[0]['change_info']['author']

            need_update = False
            if len(nanny_releases) > 0:
                [prev_snapshot_id] = nanny_releases[0]
                if prev_snapshot_id != current_snapshot_id:
                    send_nanny_update_report(service_id, current_snapshot_id, prev_snapshot_id, releaser)
                    need_update = True
            else:
                need_update = True

            if need_update:
                args = [service_id, current_snapshot_id]
                pg_query_db_and_commit('INSERT INTO nanny_releases (service_id, last_snapshot_id) VALUES (%s, %s) ' +
                                       'ON CONFLICT (service_id) DO UPDATE SET service_id = %s, last_snapshot_id = %s', args + args)

    def _check_nanny_service_statuses(self):
        nanny_client = NannyClient(token=self._token)
        services = nanny_client.get_services()

        for service_id in [x['_id'] for x in services]:
            if 'prod' not in service_id:
                continue

            nanny_client.check_pause_state(service_id)
