from travel.hotels.devops.slack_forwarder.app import app  # should be the first import to call monkey.patch_all() early enough

import requests
import datetime
import random
import re
import time
import pytz
from queue import Queue
from enum import Enum
from typing import List

from travel.hotels.devops.slack_forwarder import utils
from travel.hotels.devops.slack_forwarder.wiki_api import WikiApi
from travel.hotels.devops.slack_forwarder.pg_db import pg_query_db, pg_query_db_and_commit
from travel.hotels.devops.slack_forwarder.lost_commits_data import LostCommitsReport, LostCommitsInfoByAuthor, LostCommitsInfo

EMOJIS_MAPPING = {  # You may not see some of them, but they are here
    ':airplane:': '✈️',
    ':airplane_departure:': '🛫',
    ':candy:': '🍬',
    ':floppy_disk:': '💾',
    ':grey_question:': '❔',
    ':hammer_and_pick:': '⚒️',
    ':hankey:': '💩',
    ':heavy_minus_sign:': '➖',
    ':no_entry_sign:': '🚫',
    ':rocket:': '🚀',
    ':satellite:': '📡',
    ':camera:': '📷',
    ':thinking_face:': '🤔',
    ':watch:': '⌚',
    ':x:': '❌',
    ':mag_right:': '🔎',
    ':zap:': '⚡',
    ':heavy_dollar_sign:': '💲',
    ':arrow_heading_up:': '⤴',
    ':fire:': '🔥',
    ':gear:': '⚙',
    ':credit_card:': '💳',
    ':cat:': '🐱',
    ':1234:': '🔢',
    ':hotel:': '🏨',
    ':love_hotel:': '🏩',
    ':old_key:': '🗝',
    ':moneybag:': '💰',
    ':writing_hand:': '✍️',
    ':heart:': '❤️',
    ':clown_face:': '🤡',
}


class ThreadLinkType(Enum):
    Sandbox = 0,
    Nanny = 1,


class State:
    def __init__(self, name, priority, pretty_name, emoji, env=None, thread_text=None, mention_author=False, thread_link_type=None, ready_for_release=True, released_to_stable=False):
        self.name = name
        self.priority = priority
        self.pretty_name = pretty_name
        self.emoji = emoji
        self.unicode_emoji = EMOJIS_MAPPING.get(emoji)
        self.env = env
        self.thread_text = thread_text
        self.mention_author = mention_author
        self.thread_link_type = thread_link_type
        self.ready_for_release = ready_for_release
        self.released_to_stable = released_to_stable

    def get_sb_link(self, task_id):
        return f'https://sandbox.yandex-team.ru/task/{task_id}'

    def get_nanny_link(self, task_id):
        if self.env:
            return f'https://nanny.yandex-team.ru/ui/#/r/SANDBOX_RELEASE-{task_id}-{self.env}/'
        return None

    def get_thread_message(self, component_name, task_id, committer, mute_mentions):
        link = self.get_sb_link(task_id) if self.thread_link_type == ThreadLinkType.Sandbox else self.get_nanny_link(task_id)
        message = f"*{component_name}*: <{link}|{self.thread_text}>"
        if not mute_mentions and self.mention_author:
            message += f' @{committer}'
        return message


class States:
    Unknown = State('Unknown', priority=0, pretty_name="Unknown", emoji=":grey_question:", ready_for_release=False)
    Created = State('Created', priority=1, pretty_name="Created task", emoji=":watch:", ready_for_release=False)
    Enqueued = State('Enqueued', priority=2, pretty_name="Enqueued", emoji=":watch:", ready_for_release=False)
    Analyzing = State('Analyzing', priority=3, pretty_name="Analyzing changes", emoji=":thinking_face:", ready_for_release=False)
    Started = State('Started', priority=4, pretty_name="Building", emoji=":hammer_and_pick:", ready_for_release=False)
    Succeed = State('Succeed', priority=5, pretty_name="Build succeed", emoji=":floppy_disk:")
    Failed = State('Failed', priority=6, pretty_name="Build failed", emoji=":x:", ready_for_release=False,
                   thread_text='Сборка упала', mention_author=True, thread_link_type=ThreadLinkType.Sandbox)

    ReleasingToUnstable = State('ReleasingToUnstable', priority=7, pretty_name="Releasing to unstable", emoji=":hankey:")
    ReleasedToUnstable = State('ReleasedToUnstable', priority=8, pretty_name="Resource released to unstable", emoji=":hankey:",
                               thread_text='Релиз ресурсов в unstable завершен', thread_link_type=ThreadLinkType.Sandbox)
    DeployingToUnstable = State('DeployingToUnstable', priority=9, pretty_name="Deploying to unstable", emoji=":hankey:", env='UNSTABLE')
    DeployedToUnstable = State('DeployedToUnstable', priority=10, pretty_name="Deployed to unstable", emoji=":candy:", env='UNSTABLE',
                               thread_text='Деплой в unstable завершен', thread_link_type=ThreadLinkType.Nanny)
    DeployToUnstableFailed = State('DeployToUnstableFailed', priority=11, pretty_name="Deploy to unstable failed", emoji=":heavy_minus_sign:", env='UNSTABLE',
                                   thread_text='Деплой в unstable упал', mention_author=True, thread_link_type=ThreadLinkType.Nanny)
    DeployToUnstableCancelled = State('DeployToUnstableCancelled', priority=12, pretty_name="Deploy to unstable cancelled", emoji=":no_entry_sign:", env='UNSTABLE',
                                      thread_text='Деплой в unstable отменен', thread_link_type=ThreadLinkType.Nanny)

    ReleasingToTesting = State('ReleasingToTesting', priority=13, pretty_name="Releasing to testing", emoji=":airplane_departure:")
    ReleasedToTesting = State('ReleasedToTesting', priority=14, pretty_name="Resource released to testing", emoji=":airplane_departure:",
                              thread_text='Релиз ресурсов в тестинг завершен', thread_link_type=ThreadLinkType.Sandbox)
    DeployingToTesting = State('DeployingToTesting', priority=15, pretty_name="Deploying to testing", emoji=":airplane_departure:", env='TESTING')
    DeployedToTesting = State('DeployedToTesting', priority=16, pretty_name="Deployed to testing", emoji=":airplane:", env='TESTING',
                              thread_text='Деплой в тестинг завершен', thread_link_type=ThreadLinkType.Nanny)
    DeployToTestingFailed = State('DeployToTestingFailed', priority=17, pretty_name="Deploy to testing failed", emoji=":heavy_minus_sign:", env='TESTING',
                                  thread_text='Деплой в тестинг упал', mention_author=True, thread_link_type=ThreadLinkType.Nanny)
    DeployToTestingCancelled = State('DeployToTestingCancelled', priority=18, pretty_name="Deploy to testing cancelled", emoji=":no_entry_sign:", env='TESTING',
                                     thread_text='Деплой в тестинг отменен', thread_link_type=ThreadLinkType.Nanny)

    BuildDisabled = State('BuildDisabled', priority=19, pretty_name="Build disabled", emoji=":no_entry_sign:",
                          thread_text='Сборка пропущена', thread_link_type=ThreadLinkType.Sandbox, ready_for_release=False)
    BuildFiltered = State('BuildFiltered', priority=20, pretty_name="Build filtered", emoji=":heavy_minus_sign:",
                          thread_text='Сборка пропущена', thread_link_type=ThreadLinkType.Sandbox, ready_for_release=False)
    ReleaseDisabled = State('ReleaseDisabled', priority=21, pretty_name="Release disabled", emoji=":no_entry_sign:",
                            thread_text='Релиз пропущен', thread_link_type=ThreadLinkType.Sandbox, ready_for_release=False)

    ReleasingToStable = State('ReleasingToStable', priority=22, pretty_name="Releasing to stable", emoji=":rocket:", released_to_stable=True)
    ReleasedToStable = State('ReleasedToStable', priority=23, pretty_name="Resource released to stable", emoji=":rocket:",
                             thread_text='Релиз ресурсов в прод завершен', thread_link_type=ThreadLinkType.Sandbox, released_to_stable=True)
    DeployingToStable = State('DeployingToStable', priority=24, pretty_name="Deploying to stable", emoji=":rocket:", env='STABLE', released_to_stable=True)
    DeployedToStable = State('DeployedToStable', priority=25, pretty_name="Deployed to stable", emoji=":satellite:", env='STABLE',
                             thread_text='Деплой в прод завершен', thread_link_type=ThreadLinkType.Nanny, released_to_stable=True)
    DeployToStableFailed = State('DeployToStableFailed', priority=26, pretty_name="Deploy to stable failed", emoji=":heavy_minus_sign:", env='STABLE',
                                 thread_text='Деплой в прод упал', mention_author=True, thread_link_type=ThreadLinkType.Nanny, released_to_stable=True)
    DeployToStableCancelled = State('DeployToStableCancelled', priority=27, pretty_name="Deploy to stable cancelled", emoji=":no_entry_sign:", env='STABLE',
                                    thread_text='Деплой в прод отменен', thread_link_type=ThreadLinkType.Nanny, released_to_stable=True)


def get_pretty_component_name(name, emoji=True, unicode_emoji=False):
    mapping = {
        'searcher': (':mag_right:', 'Searcher'),
        'offercache': (':zap:', 'Offercache'),
        'pricechecker': (':heavy_dollar_sign:', 'PriceChecker'),
        'redir': (':arrow_heading_up:', 'Redir'),
        'boiler': (':fire:', 'Boiler'),
        'api': (':gear:', 'API'),
        'slack-forwarder': (':slack:', 'Slack Forwarder'),
        'orders': (':credit_card:', 'Orders'),
        'boy_catroom_builder': (':cat:', 'BoY Catroom Builder'),
        'geocounter': (':1234:', 'GeoCounter'),
        'hotels-administrator': (':hotel:', 'Hotels Administrator'),
        'hotels-extranet': (':love_hotel:', 'Extranet'),
        'external-api': (':old_key:', 'External API'),
        'teller-app': (':moneybag:', 'Teller'),
        'room-photos-uploader': (':camera:', 'Room photos uploader'),
        'busbroker': (':writing_hand:', 'Busbroker'),
        'tugc': (':heart:', 'TUGC'),
        'lototron': (':clown_face:', 'Lototron'),
    }
    if name in mapping:
        emoji_part = None
        if emoji:
            if unicode_emoji:
                emoji_part = EMOJIS_MAPPING.get(mapping[name][0])
            else:
                emoji_part = mapping[name][0]
        return f'{emoji_part} {mapping[name][1]}' if emoji_part is not None else mapping[name][1]
    return name


def get_pretty_component_state(state, task_id):
    sb_link = f'https://sandbox.yandex-team.ru/task/{task_id}'
    nanny_link = state.get_nanny_link(task_id)
    if nanny_link:
        return f'<{sb_link}|{state.pretty_name}> {state.emoji} (<{nanny_link}|nanny>)'
    return f'<{sb_link}|{state.pretty_name}> {state.emoji}'


def split_commit_message(message):
    line_break = message.find('\n')
    if line_break != -1:
        return message[:line_break].strip(), message[line_break:].strip()
    if len(message) > 150:
        return message[:150] + "...", "..." + message[150:]
    return message, ""


def get_component_name_by_nanny_service_id(nanny_service_id):
    for k, v in app.config['NANNY_COMPONENTS'].items():
        if nanny_service_id in v:
            return k
    return nanny_service_id


class ComponentInfo:
    def __init__(self, state, task_id):
        self.state = state
        self.task_id = task_id


class SlackSender:
    def __init__(self):
        self.channels = app.config['SLACK_CHANNELS']
        self.api_key = app.config['SLACK_API_KEY']

    def send(self, message):
        return self._send(message, "chat.postMessage")

    def update(self, message):
        return self._send(message, "chat.update")

    def send_plaintext(self, text, channel_name):
        return self._send(self._build_message(text, channel_name), "chat.postMessage")

    def _build_message(self, text, channel_name):
        return {
            "channel": self.channels[channel_name],
            "text": text,
            "link_names": 1,
        }

    def _send(self, message, url):
        try:
            url = f'https://slack.com/api/{url}'
            headers = {
                "Authorization": "Bearer {}".format(self.api_key),
                "Content-Type": "application/json; charset=utf-8"
            }
            app.logger.info("Will send '{}' to {}".format(message, url))
            resp = requests.post(url, headers=headers, json=message)
            app.logger.info(f"slack returned status code {resp.status_code}, reply is '{resp.text}'")
            return resp.json()
        except Exception:
            app.logger.exception("Unable to send notification")


class SlackLostCommitsReporter:
    def __init__(self):
        self.channel = app.config['SLACK_CHANNELS']['devops']
        self.sender = SlackSender()
        self.max_blocks = 50

    def send_lost_commits_report(self, report: LostCommitsReport):
        resp = self.sender.send(self._build_card(report.by_components))
        ts = resp.get('ts')
        for author, infos in report.by_authors.items():
            self.sender.send(self._build_thread_message(ts, author, infos))

    def _build_card(self, report: List[LostCommitsInfo]):
        blocks = [{
            "type": "section",
            "text": {
                "type": "mrkdwn",
                "text": f"Некоторые компоненты давно не релизились (всего {len(report)}):"
            }
        }]
        if len(report) + 1 > self.max_blocks:
            report = report[:self.max_blocks - 1]
        allow_dividers = len(report) * 3 + 1 <= self.max_blocks
        allow_footers = len(report) * 2 + 1 <= self.max_blocks

        now_ts = int(time.time())
        for item in report:
            pretty_component = get_pretty_component_name(item.component_name)
            oldest_commit_ts = min([x.timestamp for x in item.commits])
            max_days = int((now_ts - oldest_commit_ts) / (60 * 60 * 24))
            if allow_dividers:
                blocks.append({"type": "divider"})
            blocks.append({
                "type": "section",
                "text": {
                    "type": "mrkdwn",
                    "text": f"*{pretty_component}* - {self._get_commits_form(len(item.commits))} релиза (старейшему из них {max_days} {utils.get_days_form(max_days)})"
                }
            })
            if allow_footers:
                max_items = 9
                texts = ['<https://a.yandex-team.ru/arc/commit/{0}|r{0}>'.format(commit.revision) for commit in
                         item.commits[:max_items - 1]]
                if len(item.commits) > max_items:
                    texts.append(f'и ещё {len(item.commits) - max_items}')
                blocks.append({
                    "type": "context",
                    "elements": [{
                        "type": "mrkdwn",
                        "text": text
                    } for text in texts]
                })
        return {
            "channel": self.channel,
            "text": 'Lost commits report',
            "blocks": blocks
        }

    def _build_thread_message(self, ts: str, author: str, infos: List[LostCommitsInfoByAuthor]):
        text_lines = [f'@{author}, в следующих компонентах есть твои старые коммиты:']
        for info in infos:
            pretty_component = get_pretty_component_name(info.component_name, emoji=False)
            max_items = 5
            commit_texts = ['<https://a.yandex-team.ru/arc/commit/{0}|r{0}>'.format(commit.revision) for commit in
                            info.commits[:max_items - 1]]
            if len(info.commits) > max_items:
                commit_texts.append(f'и ещё {len(info.commits) - max_items}')
            commits_text = ' '.join(commit_texts)
            text_lines.append(f'{pretty_component}: {commits_text}')
        return {
            "channel": self.channel,
            "text": '\n'.join(text_lines),
            "link_names": 1,
            "thread_ts": ts,
        }

    def _get_commits_form(self, commits):
        if commits == 1:
            return f"{commits} коммит ожидает"
        if (commits < 10 or commits > 20) and commits % 10 in [2, 3, 4]:
            return f"{commits} коммита ожидают"
        return f"{commits} коммитов ожидают"


class SlackReleaseReporter:
    def __init__(self):
        self.devops_channel = app.config['SLACK_CHANNELS']['devops']
        self.release_channel = app.config['SLACK_CHANNELS']['travel-release']
        self.sender = SlackSender()
        self.ts = None
        self.max_blocks = 50

    def send_release_report(self, component_name, revision, task_id, releaser, revisions, wiki_path, full, to_devops):
        resp = self.sender.send(self._build_card(component_name, revision, task_id, releaser, revisions, wiki_path, full, to_devops))
        self.ts = resp.get('ts')
        return self.ts

    def _build_card(self, component_name, revision, task_id, releaser, revisions, wiki_path, full, to_devops):
        pretty_component = get_pretty_component_name(component_name)
        channel = self.devops_channel if to_devops else self.release_channel
        emojis = [':bomb:', ':boom:', ':crown:', ':gem:', ':tada:', ':confetti_ball:', ':medal:', ':sparkles:',
                  ':rotating_light:', ':flying_saucer:', ':rainbow:', ':radioactive_sign:', ':100:', ':cool:']
        allow_dividers = len(revisions) * 3 + 2 <= self.max_blocks
        allow_footers = len(revisions) * 2 + 2 <= self.max_blocks
        allow_blocks = len(revisions) + 2 <= self.max_blocks
        blocks = [{
            "type": "section",
            "text": {
                "type": "mrkdwn",
                "text": "*{0}* is releasing {1}".format(pretty_component, random.choice(emojis))
            }
        }, {
            "type": "context",
            "elements": [
                {
                    "type": "mrkdwn",
                    "text": "<https://staff.yandex-team.ru/{0}|By {0}@>".format(releaser)
                },
                {
                    "type": "mrkdwn",
                    "text": "<https://a.yandex-team.ru/arc/commit/{0}|r{0}>".format(revision)
                },
                {
                    "type": "mrkdwn",
                    "text": "<https://sandbox.yandex-team.ru/task/{0}|sb:{0}>".format(task_id)
                },
                {
                    "type": "mrkdwn",
                    "text": "<https://wiki.yandex-team.ru/{0}|changelog>".format(wiki_path)
                },
            ] + ([
                {
                    "type": "mrkdwn",
                    "text": "<{0}|nanny>".format(States.DeployingToStable.get_nanny_link(task_id))  # todo: avoid using explicit state
                }
            ] if component_name in app.config['NANNY_COMPONENTS'] else [])
        }]
        if full:
            single_block_texts = []
            for i in range(len(revisions)):
                if allow_dividers:
                    blocks.append({"type": "divider"})
                data = revisions[i]
                revision = data['revision']
                author = data['author']
                date = datetime.datetime.strftime(datetime.datetime.strptime(data['date'], '%Y-%m-%dT%H:%M:%S%z'), '%Y-%m-%d %H:%M:%S (%Z)')
                commit_message_title, _ = split_commit_message(data['message'])
                text = "<https://a.yandex-team.ru/arc/commit/{0}|{0}> *{1}*".format(revision, commit_message_title)
                if allow_blocks:
                    blocks.append({
                        "type": "section",
                        "text": {
                            "type": "mrkdwn",
                            "text": text
                        }
                    })
                else:
                    single_block_texts.append(text)
                if allow_footers:
                    blocks.append({
                        "type": "context",
                        "elements": [
                            {
                                "type": "mrkdwn",
                                "text": "<https://staff.yandex-team.ru/{0}|{0}@>".format(author)
                            },
                            {
                                "type": "mrkdwn",
                                "text": date
                            }
                        ]
                    })
            if not allow_blocks:
                blocks.append({
                    "type": "section",
                    "text": {
                        "type": "mrkdwn",
                        "text": '\n'.join(single_block_texts)
                    }
                })
        return {
            "channel": channel,
            "text": f'Release of {pretty_component}',
            "blocks": blocks
        }


class SlackNannyUpdateReporter:
    def __init__(self):
        self.devops_channel = app.config['SLACK_CHANNELS']['devops']
        self.sender = SlackSender()
        self.ts = None

    def send_update_report(self, component_name, service_id, current_snapshot_id, prev_snapshot_id, releaser):
        resp = self.sender.send(self._build_card(component_name, service_id, current_snapshot_id, prev_snapshot_id, releaser))
        self.ts = resp.get('ts')
        return self.ts

    def _build_card(self, component_name, service_id, current_snapshot_id, prev_snapshot_id, releaser):
        pretty_component = get_pretty_component_name(component_name)
        blocks = [{
            "type": "section",
            "text": {
                "type": "mrkdwn",
                "text": "*{0}* updated in nanny".format(pretty_component)
            }
        }, {
            "type": "context",
            "elements": [
                {
                    "type": "mrkdwn",
                    "text": "<https://staff.yandex-team.ru/{0}|By {0}@>".format(releaser)
                },
                {
                    "type": "mrkdwn",
                    "text": "<https://nanny.yandex-team.ru/ui/#/services/catalog/{0}/runtime_attrs_history/{1}/diff/{2}|nanny:diff>".format(service_id, current_snapshot_id, prev_snapshot_id)
                },
                {
                    "type": "mrkdwn",
                    "text": "<https://nanny.yandex-team.ru/ui/#/services/catalog/{0}|nanny:service>".format(service_id)
                },
            ]
        }]
        return {
            "channel": self.devops_channel,
            "text": f'{pretty_component} updated in nanny',
            "blocks": blocks
        }


class SlackNannyPauseReporter:
    def __init__(self):
        self.devops_channel = app.config['SLACK_CHANNELS']['devops']
        self.sender = SlackSender()
        self.ts = None

    def send_pause_report(self, component_name, service_id, is_paused, comment, author):
        resp = self.sender.send(self._build_card(component_name, service_id, is_paused, comment, author))
        self.ts = resp.get('ts')
        return self.ts

    def _build_card(self, component_name, service_id, is_paused, comment, author):
        pretty_component = get_pretty_component_name(component_name)

        if is_paused:
            action = 'paused'
            text = ":pause_button: *{0}* paused in nanny.\nComment: '{1}'".format(pretty_component, comment)
        else:
            action = 'resumed'
            text = ":play_button: *{0}* resumed in nanny.".format(pretty_component)

        blocks = [{
            "type": "section",
            "text": {
                "type": "mrkdwn",
                "text": text,
            }
        }, {
            "type": "context",
            "elements": [
                {
                    "type": "mrkdwn",
                    "text": "<https://staff.yandex-team.ru/{0}|By {0}@>".format(author)
                },
                {
                    "type": "mrkdwn",
                    "text": "<https://nanny.yandex-team.ru/ui/#/services/catalog/{0}|nanny:service>".format(service_id)
                },
            ]
        }]
        return {
            "channel": self.devops_channel,
            "text": f'{pretty_component} {action} in nanny',
            "blocks": blocks
        }


class WikiReleaseReporter:
    def __init__(self):
        self.base_path: str = app.config['CHANGELOGS_WIKI_PATH']
        self.wiki_api: WikiApi = WikiApi(app.config['WIKI_TOKEN'])

    def send_release_report(self, component_name, revision, task_id, releaser, revisions):
        normalized_component_name = utils.normalize_component_name(component_name)
        self.wiki_api.post_page(f'{self.base_path}/{normalized_component_name}', component_name, '{{tree}}')
        path = f'{self.base_path}/{normalized_component_name}/{revision}'
        title, body = self._build_changelog(component_name, revision, task_id, releaser, revisions)
        app.logger.info(f'Posting release changelog to {path}')
        self.wiki_api.post_page(path, title, body)
        return path

    def _build_changelog(self, component_name, revision, task_id, releaser, revisions):
        pretty_component = get_pretty_component_name(component_name, emoji=False)
        time = datetime.datetime.strftime(datetime.datetime.now(tz=pytz.timezone('Europe/Moscow')), '%Y-%m-%d %H:%M (%Z)')
        title = f'Релиз {pretty_component} от {time} (r{revision})'
        timezone = datetime.datetime.strftime(datetime.datetime.strptime(revisions[0]['date'], '%Y-%m-%dT%H:%M:%S%z'),
                                              ' (%Z)') if len(revisions) > 0 else ''
        body_lines = [
            f'Компонент: {pretty_component}',
            f'Резил запустил: {releaser}@',
            f'Ревизия: ((https://a.yandex-team.ru/arc/commit/{revision} r{revision}))',
            f'Sanbox task: ((https://sandbox.yandex-team.ru/task/{task_id} {task_id}))',
            '',
            'Changelog:',
            '',
            '#|',
            f'|| Ревизия | Текст коммита | Автор | Время{timezone} | Тикеты ||',
        ]
        for data in revisions:
            revision = data['revision']
            author = data['author']
            date = datetime.datetime.strftime(datetime.datetime.strptime(data['date'], '%Y-%m-%dT%H:%M:%S%z'),
                                              '%Y-%m-%d %H:%M:%S')
            commit_message = data['message']
            formatted_commit_message = f'\n```\n{commit_message}\n```\n' if '\n' in commit_message else f'`{commit_message}`'
            tickets = '\n'.join(re.findall('[A-Z]+-\\d+', commit_message))
            body_lines.append(
                f'|| ((https://a.yandex-team.ru/arc/commit/{revision} r{revision})) | {formatted_commit_message} | {author}@ | {date} | {tickets} ||')

        body_lines += [
            '|#',
            '',
            '//Powered by «Яндекс.Путешествия — ПОНЯбот»//'
        ]
        return title, '\n'.join(body_lines)


class Notification:
    def __init__(self, revision, committer, commit_message, ts):
        self.channel = app.config['SLACK_CHANNELS']['travel-build']
        self.sender = SlackSender()
        self.revision = revision
        self.committer = committer
        self.commit_message_title, self.commit_message_body = split_commit_message(commit_message)
        self.ts = ts

    def send_or_update_notification(self, components):
        if self.ts:
            self.sender.update(self._build_card(components))
        else:
            resp = self.sender.send(self._build_card(components))
            self.ts = resp.get('ts')
            return self.ts

    def send_updates_to_thread(self, component, state, task_id, committer, mute_mentions):
        app.logger.info(f'Going to send updates on ({component}, {state}, {task_id}, {committer})')
        if state.thread_text is None:
            return
        if not self.ts:
            app.logger.warning('skipping thread update because ts is unknown')
            return
        pretty_name = get_pretty_component_name(component, emoji=False)
        message = {
            "channel": self.channel,
            "text": state.get_thread_message(pretty_name, task_id, committer, mute_mentions),
            "link_names": 1,
            "thread_ts": self.ts,
        }
        self.sender.send(message)

    def _build_card(self, components):
        message = {
            "channel": self.channel,
            "attachments": [{
                "fallback": "Processing new revision: {}".format(self.revision),
                "color": self._get_color(components),
                "author_name": f"{self.committer}@",
                "author_link": f"https://staff.yandex-team.ru/{self.committer}",
                "title": "{0} (<https://a.yandex-team.ru/arc/commit/{1}|#{1}>)".format(self.commit_message_title, self.revision),
                "text": self.commit_message_body,
                "footer": "Яндекс.Путешествия — автосборка",
                "fields": [{
                    "title": get_pretty_component_name(name),
                    "value": get_pretty_component_state(component.state, component.task_id),
                    "short": True
                } for name, component in components.items() if component.state != States.BuildFiltered],
            }]
        }
        if self.ts:
            message['ts'] = self.ts
        return message

    def _get_color(self, components):
        if all([self._is_ok_state(x.state) for x in components.values()]):
            return 'good'
        if any([self._is_fail_state(x.state) for x in components.values()]):
            return 'danger'
        return 'no'

    def _is_ok_state(self, state):
        return state in [States.ReleasedToTesting, States.ReleasedToStable, States.ReleaseDisabled, States.BuildDisabled]

    def _is_fail_state(self, state):
        return state in [States.Failed]


class Notifier:
    def __init__(self):
        self._queue = Queue()

    def run_sender(self):
        with app.app_context():
            while True:
                try:
                    revision = self._queue.get()
                    self._post_notification(revision)
                except Exception:
                    app.logger.error('Exception while posting notification', exc_info=True)

    def enqueue_notification(self, revision):
        self._queue.put(revision)

    def send_plaintext_notification(self, text, channel_name):
        sender = SlackSender()
        sender.send_plaintext(text, channel_name)

    def send_release_report(self, component_name, revision, task_id, releaser, revisions):
        wiki_path = None
        try:
            wiki_release_reporter = WikiReleaseReporter()
            wiki_path = wiki_release_reporter.send_release_report(component_name, revision, task_id, releaser, revisions)
        except Exception:
            app.logger.error('Exception while posting release report to wiki', exc_info=True)

        try:
            if component_name in app.config['COMPONENTS_POSTED_TO_DEVOPS']:
                SlackReleaseReporter().send_release_report(component_name, revision, task_id, releaser, revisions, wiki_path, full=False, to_devops=True)
            SlackReleaseReporter().send_release_report(component_name, revision, task_id, releaser, revisions, wiki_path, full=True, to_devops=False)
        except Exception:
            app.logger.error('Exception while posting release report to slack', exc_info=True)

    def _post_notification(self, revision):
        events = pg_query_db('SELECT component, event_type, task_id FROM events WHERE revision = %s', [str(revision)])
        builds = pg_query_db('SELECT committer, commit_message, ts FROM builds WHERE revision = %s', [str(revision)])
        if len(builds) == 0:
            app.logger.warn(f'No registered build for revision {revision}')
            return
        (committer, commit_message, ts) = builds[-1]
        app.logger.info(f'Events for revision {revision}: {events}')
        app.logger.info(f'Info for revision {revision}: {(committer, commit_message, ts)}')
        components = {}
        for component, event_type, task_id in events:
            if component not in components or components[component].state.priority < getattr(States, event_type).priority:
                components[component] = ComponentInfo(getattr(States, event_type), task_id)
        app.logger.info(f'Components: {components}')

        notification = Notification(revision, committer, commit_message, ts)
        ts = notification.send_or_update_notification(components)
        if ts:
            pg_query_db_and_commit('UPDATE builds SET ts = %s WHERE revision = %s', [ts, str(revision)])

        processed_events = pg_query_db('SELECT component, event_type, task_id FROM events WHERE revision = %s AND processed = 1', [str(revision)])
        seen_events = {(component, event_type, task_id) for component, event_type, task_id in processed_events}

        not_processed_events = pg_query_db('SELECT id, component, event_type, task_id FROM events WHERE revision = %s AND processed = 0', [str(revision)])

        for id, component, event_type, task_id in not_processed_events:
            mute_mentions = (component, event_type, task_id) in seen_events
            notification.send_updates_to_thread(component, getattr(States, event_type), task_id, committer, mute_mentions)
            pg_query_db_and_commit('UPDATE events SET processed = 1 WHERE id = %s', [id])

        pg_query_db_and_commit("DELETE FROM builds WHERE timestamp < (NOW() - INTERVAL '60 DAY')", [])
        pg_query_db_and_commit("DELETE FROM events WHERE timestamp < (NOW() - INTERVAL '60 DAY')", [])
        pg_query_db_and_commit("DELETE FROM deployments WHERE timestamp < (NOW() - INTERVAL '60 DAY')", [])

    def send_nanny_update_report(self, nanny_service_id, current_snapshot_id, prev_snapshot_id, releaser):
        component_name = get_component_name_by_nanny_service_id(nanny_service_id)
        try:
            slack_nanny_release_reporter = SlackNannyUpdateReporter()
            slack_nanny_release_reporter.send_update_report(component_name, nanny_service_id, current_snapshot_id, prev_snapshot_id, releaser)
        except Exception:
            app.logger.error('Exception while posting nanny release report to slack', exc_info=True)

    def send_pause_report(self, service_id, is_paused, comment, author):
        component_name = get_component_name_by_nanny_service_id(service_id)
        try:
            slack_nanny_pause_reporter = SlackNannyPauseReporter()
            slack_nanny_pause_reporter.send_pause_report(component_name, service_id, is_paused, comment, author)
        except Exception:
            app.logger.error('Exception while posting nanny pause report to slack', exc_info=True)
