from travel.hotels.devops.slack_forwarder.app import app  # should be the first import to call monkey.patch_all() early enough

import json
import gevent
import requests
import tvm2
import ticket_parser2.api.v1 as tp2
from flask import request, render_template, jsonify, url_for, redirect
from gevent.pywsgi import WSGIServer
from blackbox import JsonBlackbox, BaseBlackbox
from logging.config import dictConfig

from travel.hotels.devops.slack_forwarder.birthdays import BirthdaysReporter
from travel.hotels.devops.slack_forwarder.component_info_builder import ComponentInfoBuilder
from travel.hotels.devops.slack_forwarder.pg_db import pg_init_db, pg_query_db, pg_query_db_and_commit
from travel.hotels.devops.slack_forwarder.db_locks import get_pg_lock_and_wait
from travel.hotels.devops.slack_forwarder.lost_commits_watcher import LostCommitsWatcher
from travel.hotels.devops.slack_forwarder.nanny import NannyWatcher, NannyClient
from travel.hotels.devops.slack_forwarder.notifier import States
from travel.hotels.devops.slack_forwarder.notifier_worker import run as run_notifier_worker
from travel.hotels.devops.slack_forwarder.notifier_worker import trigger_notification, send_release_report
from travel.hotels.devops.slack_forwarder.onduty import OnDuty
from travel.hotels.devops.slack_forwarder.st_duty import StDuty
from travel.hotels.devops.slack_forwarder.st_duty_auto_actions import StDutyAutoActions
from travel.hotels.devops.slack_forwarder.sandbox import SbClient, ReleaseFailedException, ReleaseBadRequestException
from travel.hotels.devops.slack_forwarder.blackbox_middleware import blackbox, NeedRedirectError

PATH_MAPPINGS = {
    "/travel-build": "/services/T1B8NLW31/BCF162207/4udMDGFHI5cAP1gZq2JaWfBk",
    "/travel-utils": "/services/T1B8NLW31/BDPK7LDAA/xXIx6UqIj0XgkhXi8smIqOgM",
    "/test-alittleprince": "/services/T1B8NLW31/BDNRF7Z0B/ZFvgaxqGP0DKti1q9idfK70v",
    "/test-tivelkov": "/services/T1B8NLW31/BCBJX8PHQ/iBlfjo4FVutPfjtqgCcc5dRU",  # PM
    "/travel-devops": "/services/T1B8NLW31/BDRUWGLQN/sQMZ80lGawvVaScbmzuY3Nvs",
    "/travel-orders": "/services/T1B8NLW31/B01BW7PACM8/HglurRNFGNc7ggD9WclxbYho",
    "/travel-mpivko": "/services/T1B8NLW31/BLZK7F8JK/wRWIbLpRW0ddbYCP3ETZ5PeG"
}

nanny_watcher = None
component_info_builder: ComponentInfoBuilder = None
sb_client: SbClient = None
bb_client: JsonBlackbox= None
tvm2_client: tvm2.TVM2 = None
wsgi_server: WSGIServer = None


def init_logging():
    dictConfig({
        'version': 1,
        'formatters': {'default': {
            'format': '[%(asctime)s] %(levelname)s in %(module)s: %(message)s',
        }},
        'handlers': {
            'wsgi': {
                'class': 'logging.StreamHandler',
                'stream': 'ext://flask.logging.wsgi_errors_stream',
                'formatter': 'default',
            },
            'file': {
                'class': 'logging.handlers.WatchedFileHandler',
                'formatter': 'default',
                'filename': app.config['LOG_FILENAME'],
            },
        },
        'root': {
            'level': 'INFO',
            'handlers': ['wsgi', 'file']
        }
    })


def _do_run():
    global nanny_watcher
    global component_info_builder
    global sb_client
    global tvm2_client
    global bb_client

    gevent.spawn(run_notifier_worker)
    sb_client = SbClient(app.config['SANDBOX_TOKEN'])
    tvm2_client = tvm2.TVM2(
        client_id=app.config['SELF_TVM_ID'],
        blackbox_client=tp2.BlackboxClientId.ProdYateam,
        secret=app.config['TVM_SECRET']
    )
    bb_client = JsonBlackbox(
        url=BaseBlackbox.URLS['intranet']['production'],
        tvm2_client_id=app.config['SELF_TVM_ID'],
        tvm2_secret=app.config['TVM_SECRET'],
        blackbox_client=tp2.BlackboxClientId.ProdYateam
    )

    component_info_builder = ComponentInfoBuilder(app.config['ARC_TOKEN'])
    nanny_watcher = NannyWatcher(app.config['NANNY_TOKEN'])
    components_to_run = [
        component_info_builder,
        BirthdaysReporter(app.config['STAFF_TOKEN']),
        nanny_watcher,
        OnDuty(app.config['ABC_TOKEN']),
        LostCommitsWatcher(app.config['ARC_TOKEN']),
        StDuty(app.config['ABC_TOKEN'], app.config['ST_TOKEN']),
        StDutyAutoActions(app.config['ST_TOKEN']),
    ]

    for component in components_to_run:
        gevent.spawn(component.run)

    host = app.config['HOST']
    port = app.config['PORT']
    app.logger.info(f'Serving on http://{host}:{port}...')

    global wsgi_server
    wsgi_server = WSGIServer((host, port), app, log=app.logger)
    wsgi_server.serve_forever()


def run():
    init_logging()
    with app.app_context():
        pg_init_db()

    global wsgi_server
    with app.app_context():
        get_pg_lock_and_wait(1, _do_run, lambda: wsgi_server.stop())


def get_bb_client():
    return bb_client


@app.route('/ping', methods=["GET"])
def ping():
    return 'OK'


@app.route('/travel-build', methods=["POST"])
@app.route('/travel-utils', methods=["POST"])
@app.route('/test-alittleprince', methods=["POST"])
@app.route('/test-tivelkov', methods=["POST"])
@app.route('/travel-devops', methods=["POST"])
@app.route('/travel-orders', methods=["POST"])
@app.route('/travel-mpivko', methods=["POST"])
def incoming_webhook_request():
    path = PATH_MAPPINGS[request.path]
    data = request.get_json(force=True)  # avoid content type check, because orders are sent with wrong content type :(
    app.logger.info("Data to transfer is '{}'".format(data))
    resp = requests.post(f'https://hooks.slack.com/{path}', json=data)
    if not resp.ok:
        app.logger.error(f'Failed to send incoming webhook (status: {resp.status_code}): {resp.content}')
        return "FAIL", {'Content-type': 'text/plain'}
    return "SENT", {'Content-type': 'text/plain'}


@app.route('/build-notifications/register-build', methods=["POST"])
def register_build():
    data = request.json
    app.logger.info(f'Build registration: {data}')
    args = [str(data['revision']), data['committer'], data['commit_message']]
    pg_query_db_and_commit('INSERT INTO builds (revision, committer, commit_message) VALUES (%s, %s, %s) ' +
                           'ON CONFLICT (revision) DO UPDATE SET revision = %s, committer = %s, commit_message = %s', args + args)
    return ''


@app.route('/build-notifications/report-status', methods=["POST"])
def report_status():
    data = request.json
    app.logger.info(f'Status report: {data}')
    if not hasattr(States, data['status']):
        return f'Invalid status: {data["status"]}', 400
    pg_query_db_and_commit('INSERT INTO events (revision, component, event_type, task_id) VALUES (%s, %s, %s, %s)',
                           [data['revision'], data['component_name'], data['status'], data['task_id']])
    trigger_notification(data['revision'])
    return ''


@app.route('/build-notifications/report-release', methods=["POST"])
def report_release():
    data = request.json
    app.logger.info(f'Status report: {data}')

    revision = data['revision']
    component_name = data['component_name']
    revisions = data['revisions']
    task_id = data['task_id']
    releaser = data.get('releaser', 'unknown')
    env = data['env']

    revisions = sorted(revisions, key=lambda x: int(x['revision']), reverse=True)

    pg_query_db_and_commit('INSERT INTO releases (revision, component_name, revisions, task_id, releaser, env) VALUES (%s, %s, %s, %s, %s, %s)',
                           [revision, component_name, json.dumps(revisions), task_id, releaser, env])

    send_release_report(component_name, revision, task_id, releaser, revisions)

    return ''


@app.route('/monitoring/events', methods=["GET"])
def pg_monitoring_events():
    events = pg_query_db('SELECT * FROM events', [])
    return '<br/>'.join([json.dumps(event) for event in events])


@app.route('/monitoring/builds', methods=["GET"])
def pg_monitoring_builds():
    builds = pg_query_db('SELECT * FROM builds', [])
    return '<br/>'.join([json.dumps(build) for build in builds])


@app.route('/monitoring/deployments', methods=["GET"])
def pg_monitoring_deployments():
    deployments = pg_query_db('SELECT * FROM deployments', [])
    return '<br/>'.join([json.dumps(deployment) for deployment in deployments])


@app.route('/force-send-overrides-stats', methods=["GET"])
def force_send_overrides_stats():
    global nanny_watcher
    nanny_watcher._check_override_resources(force=True)
    return 'OK'


@app.route('/force-process-not-processed-events', methods=["GET"])
def force_process_not_processed_events():
    events = pg_query_db('SELECT id, component, event_type, task_id, revision FROM events WHERE processed = 0', [])
    for id, component, event_type, task_id, revision in events:
        trigger_notification(revision)
    return '<br/>'.join(['Triggered notification for:'] + [json.dumps(event) for event in events])


@app.route('/', methods=["GET"])
def root():
    return redirect(url_for('get_components'), code=302)


@app.route('/components', methods=["GET"])
@blackbox(get_bb_client)
def get_components():
    return render_template('components.html', component_infos=component_info_builder.get_components_info())


@app.route('/component/<component_name>', methods=["GET"])
@blackbox(get_bb_client)
def get_component(component_name):
    return render_template('component.html',
                           component_name=component_info_builder.get_pretty_component_name(component_name, emoji=False),
                           commits=component_info_builder.get_component_commits(component_name))


@app.route('/api/release', methods=["POST"])
@blackbox(get_bb_client)
def release():
    data = request.json
    app.logger.info(f'Release request: {data}')
    task_id = data['task_id']
    component, revision = pg_query_db('SELECT component, revision FROM events WHERE task_id=%s LIMIT 1', [task_id])[0]

    service_tickets = tvm2_client.get_service_tickets(app.config['SANDBOX_TVM_ID'])
    try:
        sb_client.release_task_to_stable(task_id=task_id, service_ticket=service_tickets['2002826'])
        result = {
            'status': 'ok',
            'revision': revision,
            'task_id': task_id,
        }
        if component in app.config['NANNY_COMPONENTS']:
            result['nanny_link'] = States.DeployingToStable.get_nanny_link(task_id),  # todo: avoid using explicit state
        return jsonify(result)
    except ReleaseBadRequestException as e:
        app.logger.warn(f'Bad request on release: {e.error_text}')
        return e.error_text, 400
    except ReleaseFailedException as e:
        app.logger.error(f'Error on release: {e.error_text}')
        return e.error_text, 500
    except Exception as e:
        app.logger.error('Error on release', exc_info=True)
        return str(e), 500


@app.route('/api/pause', methods=["POST"])
@blackbox(get_bb_client)
def pause():
    data = request.json
    app.logger.info(f'Pause request: {data}')
    service_id = data['service_id']
    action = data['action']
    comment = data['comment']

    nanny_client = NannyClient(session_id=request.cookies.get('Session_id'))
    if action == 'pause':
        nanny_client.pause_service(service_id, comment)
    else:
        nanny_client.resume_service(service_id, '')
    nanny_client.check_pause_state(service_id)

    result = {
        'status': 'ok',
        'nanny_service_id': service_id,
        'result': 'paused' if action == 'pause' else 'resumed'
    }
    return jsonify(result)


@app.errorhandler(NeedRedirectError)
def handle_bb_redirect(e):
    return e.response
