import time
from typing import List, Optional, Iterable

from startrek_client import Startrek
import startrek_client.exceptions

from travel.hotels.devops.slack_forwarder.app import app
from travel.hotels.devops.slack_forwarder.abc_service import AbcService
from travel.hotels.devops.slack_forwarder.duty_registry import DutyRegistry, DutyConfig


class CachedAbcDuty:
    def __init__(self, abc_token: str, services: Iterable[DutyConfig]):
        self._abc_service = AbcService(abc_token)
        self._abc_last_sync_time = 0
        self._abc_sync_interval_sec = 10 * 60  # 10 min
        self._services = list(services)
        self._cache = {}

    def get(self, service_id) -> Optional[List[AbcService.DutyRole]]:
        if time.time() - self._abc_last_sync_time > self._abc_sync_interval_sec:
            self._abc_last_sync_time = time.time()
            self._cache = {x.service_id: self._abc_service.get_duty_roles(x) for x in self._services}
        return self._cache.get(service_id)


class StDuty:
    def __init__(self, abc_token, st_token):
        registry = DutyRegistry()
        self.tag_to_service_mapping = {x.duty_required_tag: x for x in registry.get_duties_for_st_duty()}
        self.queues = [
            'HOTELPROCESS',
            'TRAVELSUP',
            'TRAVELORDERSUP',
            'TRAVELBACK',
            'TRAINSSUP',
            'BUSSUP',
            'TRAVELACCOUNT',
            'AVIASUP',
            'TOURSSUP',
            'RASPFEEDBACK',
        ]
        self._cached_abc_duty = CachedAbcDuty(abc_token, self.tag_to_service_mapping.values())
        self._st_client = Startrek(useragent='python', base_url='https://st-api.yandex-team.ru', token=st_token)
        self._processed = dict()

    def run(self):
        with app.app_context():
            while True:
                try:
                    # We cache processed tickets for 1 hour to avoid problems when st is partially down
                    self._processed = {k: v for k, v in self._processed.items() if time.time() - v < 60 * 60}
                    for queue in self.queues:
                        for tag, duty_config in self.tag_to_service_mapping.items():
                            duty_logins = [x.user_login for x in self._cached_abc_duty.get(duty_config.service_id)]
                            issues = self._st_client.issues.find(
                                filter={'queue': queue, 'tags': tag},
                                per_page=10000
                            )
                            if len(issues) > 0:
                                app.logger.info('Need to summon duty developer to tickets: {}'.format([x['key'] for x in issues]))
                                for issue in issues:
                                    self._process_duty_ticket(issue, tag, duty_config, duty_logins)
                except Exception:
                    app.logger.error('Exception while watching st duty', exc_info=True)
                time.sleep(10)

    def _process_duty_ticket(self, issue, tag, duty_config, duty_logins):
        if tag not in self._st_client.issues[issue.key].tags:
            app.logger.info(f'Skipping notification for {issue.key}, {tag} because ticket has no key already')
            return

        if (issue.key, tag) not in self._processed:
            hotels_duty_ticket = None
            if duty_config.duty_tickets_queue is not None:
                hotels_duty_ticket = self._get_or_create_duty_ticket(duty_config.duty_tickets_queue, issue)
            comments_to_update = self._get_comments_to_update(issue.key, tag)
            if len(comments_to_update) > 0:
                self._summon_to_existing_comments(issue, tag, comments_to_update, duty_config, duty_logins, hotels_duty_ticket)
            else:
                self._summon_to_new_comment(issue, tag, duty_config, duty_logins, hotels_duty_ticket)
            self._processed[(issue.key, tag)] = time.time()
        else:
            app.logger.info(f'Skipping notification for {issue.key}, {tag} because it was already sent')
        self._st_client.issues[issue.key].update(tags={'remove': [tag], 'add': [duty_config.duty_summoneed_tag]})

    def _summon_to_existing_comments(self, issue, tag, comments_to_update, duty_config, duty_logins, hotels_duty_ticket):
        app.logger.info(f'Updating comments for {issue.key}, {tag}')
        for comment_id in comments_to_update:
            try:
                curr_comment = self._st_client.issues[issue.key].comments[comment_id]
                text = curr_comment.text + f'\n\n//Дежурный {duty_config.name_nominative_case} был призван автоматически//'
                if hotels_duty_ticket is not None:
                    text += f'\n//Для дежурного: {hotels_duty_ticket}//'
                curr_comment.update(text=text, summonees={'add': duty_logins})
            except startrek_client.exceptions.NotFound:
                pass

    def _summon_to_new_comment(self, issue, tag, duty_config, duty_logins, hotels_duty_ticket):
        app.logger.info(f'Sending notification for {issue.key}, {tag}')
        text = f'Требуется вмешательство дежурного {duty_config.name_genitive_case}.'
        if hotels_duty_ticket is not None:
            text += f' Для дежурного: {hotels_duty_ticket}'
        issue.comments.create(text=text, summonees=duty_logins)

    def _get_or_create_duty_ticket(self, queue, initial_ticket):
        links = sorted([x for x in initial_ticket.links if x.object.key.startswith(queue)], key=lambda x: x.createdAt)
        if len(links) > 0:
            if len(links) > 1:
                app.logger.warn(f'More than 1 duty tickets for {initial_ticket.key}')
            duty_ticket_key = links[0].object.key
            issue = self._st_client.issues[duty_ticket_key]
            if issue.status.key != 'open':
                to_open_transitions = [x for x in issue.transitions if x.to.key == 'open']
                if len(to_open_transitions) == 0:
                    raise Exception(f'Can\'t find transition to open state for {duty_ticket_key}')
                to_open_transitions[0].execute()
            return duty_ticket_key
        ticket = self._st_client.issues.create(queue=queue, summary=initial_ticket.summary, description=f'{initial_ticket.key}\n---\n<{{Исходное описание\n{initial_ticket.description}\n}}>')
        return ticket.key

    def _get_comments_to_update(self, issue_key, expected_tag):
        def is_tag_added_during_ticket_change(change):
            for changed_field in change['fields']:
                if changed_field['field'].id == 'tags' and expected_tag not in (changed_field['from'] or []) and expected_tag in (changed_field['to'] or []):
                    return True
            return False

        comments_to_update = []
        for change in self._st_client.issues[issue_key].changelog.get_all(sort='asc'):
            if change.comments is not None and 'added' in change.comments and is_tag_added_during_ticket_change(change):
                comments_to_update += [x.id for x in change.comments['added']]
        return comments_to_update[-1:]
