import time

from startrek_client import Startrek

from travel.hotels.devops.slack_forwarder.app import app


class StDutyAutoActions:
    def __init__(self, st_token):
        self.queues = [
            'HOTELSDUTY',  # test
            'HOTELDUTY'
        ]
        self.queues_for_notifications = [
            'HOTELPROCESS',
            'TRAVELSUP',
            'TRAVELORDERSUP',
            'TRAVELACCOUNT',
            'TRAVELBACK',
        ]
        self.on_creation_tag = 'auto_on_creation'
        self.on_clarification_request_tag = 'auto_on_clarification_request'
        self.on_reopen_tag = 'auto_on_reopen'
        self.on_close_tag = 'auto_on_close'
        self.need_info_tag = 'hotel_duty_need_info'
        self._st_client = Startrek(useragent='python', base_url='https://st-api.yandex-team.ru', token=st_token)

    def run(self):
        with app.app_context():
            while True:
                try:
                    for queue in self.queues:
                        for tag, method in [
                            (self.on_creation_tag, self._on_creation_tag),
                            (self.on_clarification_request_tag, self._on_clarification_request_tag),
                            (self.on_reopen_tag, self._on_reopen_tag),
                            (self.on_close_tag, self._on_close_tag),
                        ]:
                            issues = self._st_client.issues.find(
                                filter={'queue': queue, 'tags': tag},
                                per_page=10000
                            )
                            if len(issues) > 0:
                                app.logger.info('Handling {} in tickets: {}'.format(tag, [x['key'] for x in issues]))
                                for issue in issues:
                                    method(issue, tag)
                except Exception:
                    app.logger.error('Exception while watching st duty auto actions', exc_info=True)
                time.sleep(10)

    def _on_creation_tag(self, issue, tag):
        text = f'Создан тикет на дежурного разработчика: {issue.key}.'
        self._notify(issue, tag, text, [], [])

    def _on_clarification_request_tag(self, issue, tag):
        text = f'В тикете на дежурного требуются уточнения. Пожалуйста, обратите внимание на вопрос в тикете: {issue.key}.'
        self._notify(issue, tag, text, [], [self.need_info_tag])

    def _on_reopen_tag(self, issue, tag):
        self._notify(issue, tag, None, [self.need_info_tag], [])

    def _on_close_tag(self, issue, tag):
        text = f'Работа дежурного в связанном тикете завершена: {issue.key}.'
        self._notify(issue, tag, text, [], [])

    def _notify(self, issue, tag, text, related_remove_tags, related_add_tags):
        if tag not in self._st_client.issues[issue.key].tags:
            app.logger.info(f'Skipping notification for {issue.key}, {tag} because ticket has no key already')
            return

        related_tickets = [x.object.key for x in issue.links if x.object.queue.key in self.queues_for_notifications]
        for related_ticket in related_tickets:
            app.logger.info(f'Sending notification for {issue.key}, {tag} to {related_ticket}')
            if text is not None:
                self._st_client.issues[related_ticket].comments.create(text=text)
            self._st_client.issues[related_ticket].update(tags={'remove': related_remove_tags, 'add': related_add_tags})

        self._st_client.issues[issue.key].update(tags={'remove': [tag], 'add': []})
