#!/usr/bin/env python3

"""
This is an alternative to default run.sh generated for every JAVA_PROGRAM in Arcadia.
This alternate script allow to avoid pass full class name at script run and provides
a set of useful default paramters for JVM.
"""

import os
import sys

JAVA_PATH = "{{YA_JAVA}}"
DEBUG_PORT = 5836
BASE_LOG_DIR = "./logs"
SERVICE_NAME = "{{appName}}"
MAIN_CLASS = "{{mainClass}}"

def get_log_dir():
    log_dir = os.path.join(BASE_LOG_DIR, SERVICE_NAME)
    os.makedirs(log_dir, exist_ok=True)
    return log_dir


def get_base_dir():
    return os.path.dirname(os.path.abspath(__file__))


def get_lib_dir():
    return os.path.join(get_base_dir(), SERVICE_NAME)


def get_classpath():
    jar_dir = os.path.join(SERVICE_NAME, "*")
    return jar_dir


def get_java_args():
    return [
        "-Xrunjdwp:transport=dt_socket,server=y,suspend=n,address=%s" % DEBUG_PORT,
        "-DLOG_DIR={}".format(get_log_dir()),
        "-Djava.library.path={}".format(get_lib_dir()),

        # gc
        "-XX:MaxHeapSize={}".format(get_heap_size()),
        "-verbose:gc",
        "-XX:+PrintGC",
        "-XX:+PrintGCDetails",
        "-Xloggc:{}/gc-%t.log".format(get_log_dir()),
        "-XX:ErrorFile={}/hs_err.log".format(get_log_dir()),
        "-XX:+HeapDumpOnOutOfMemoryError",
        "-XX:HeapDumpPath={}/hprof/".format(get_log_dir()),
        "-XX:OnOutOfMemoryError=kill",
        "-Xdebug",

        # ipv6
        "-Djava.net.preferIPv6Addresses=true",
        "-Djava.net.preferIPv4Stack=false",

        # i/o
        "-Dfile.encoding=UTF-8"
    ]


def get_heap_size():
    heap_size = os.environ.get("JAVA_HEAP_SIZE")
    if heap_size is None:
        heap_size = "2G"
    return heap_size


def get_available_processors():
    cpus = os.environ.get("CPU_GUARANTEE")
    cpu_limit = os.environ.get("CPU_LIMIT")
    if cpu_limit is not None:
        cpus = cpu_limit

    if cpus is not None:
        # cpu guarantee is given in 1.989c format (for 2000 ms guarantee)
        # so we cut of the last letter and round the float number
        cpus = cpus[:-1]
        available_processors = int(round(float(cpus))) # parse-n-round
        return "-XX:ActiveProcessorCount=%s" % available_processors


if __name__ == '__main__':
    args = get_java_args()
    args.append("-classpath")
    args.append(get_classpath())
    available_processors = get_available_processors()
    if available_processors:
        args.append(available_processors)
    args.append(MAIN_CLASS)
    args.extend(sys.argv[1:])
    os.execv(JAVA_PATH, args)
