#!/usr/bin/env python

import os

JAVA_PATH = "/usr/bin/java"
JMX_REMOTE_PORT = 5834
JMX_RMI_PORT = 5835
DEBUG_PORT = 5836
BASE_LOG_DIR = "/logs"
SERVICE_NAME = "{{appName}}"
MAIN_CLASS = "{{mainClass}}"


def get_log_dir():
    return os.path.join(BASE_LOG_DIR, SERVICE_NAME)


def get_base_dir():
    return os.path.join(os.path.dirname(os.path.abspath(__file__)), "..")


def get_lib_dir():
    return os.path.join(get_base_dir(), SERVICE_NAME)


def get_classpath():
    jar_dir = os.path.join(SERVICE_NAME, "*")
    return jar_dir


def get_java_args():
    return [
        "-Dcom.sun.management.jmxremote=true",
        "-Dcom.sun.management.jmxremote.authenticate=false",
        "-Dcom.sun.management.jmxremote.ssl=false",
        "-Dcom.sun.management.jmxremote.port=%s" % JMX_REMOTE_PORT,
        "-Dcom.sun.management.jmxremote.rmi.port=%s" % JMX_RMI_PORT,
        "-Xrunjdwp:transport=dt_socket,server=y,suspend=n,address=%s" % DEBUG_PORT,
        "-Djava.rmi.server.hostname=127.0.0.1",
        "-DLOG_DIR={}".format(get_log_dir()),
        "-Djava.library.path={}".format(get_lib_dir()),

        # gc
        "-XX:MaxHeapSize={}".format(get_heap_size()),
        "-verbose:gc",
        "-XX:+PrintGC",
        "-XX:+PrintGCDetails",
        "-XX:MaxGCPauseMillis=25",
        "-Xloggc:{}/gc-%t.log".format(get_log_dir()),
        "-XX:ErrorFile={}/hs_err.log".format(get_log_dir()),
        "-XX:+HeapDumpOnOutOfMemoryError",
        "-XX:HeapDumpPath={}/hprof/".format(get_log_dir()),
        "-XX:OnOutOfMemoryError=kill -9 %p",  # It's strange, but we don't need quotes here (with quotes we get an error like "sh: 1: echo kill -9 67158: not found")
        "-Xdebug",

        # ipv6
        "-Djava.net.preferIPv6Addresses=true",
        "-Djava.net.preferIPv4Stack=false",

        # i/o
        "-Dfile.encoding=UTF-8"
    ]


def get_heap_size():
    heap_size = os.environ.get("JAVA_HEAP_SIZE")
    if heap_size is None:
        heap_size = "3G"
    return heap_size


def get_profile():
    profile = os.environ.get("a_ctype")
    if profile is not None:
        return "-Dspring.profiles.active={},overrides".format(profile)


def get_yjp():
    if os.environ.get('ENABLE_PROFILER') == '1':
        agent_path = os.environ.get('PROFILER_AGENT_PATH', '/usr/lib64/libyjpagent.so')
        return "-agentpath:{}=logdir=/dev/null".format(agent_path)


def get_available_processors():
    cpus = os.environ.get("CPU_GUARANTEE")
    cpu_limit = os.environ.get("CPU_LIMIT")
    if cpu_limit is not None:
        cpus = cpu_limit

    if cpus is not None:
        # cpu guarantee is given in 1.989c format (for 2000 ms guarantee)
        # so we cut of the last letter and round the float number
        cpus = cpus[:-1]
        available_processors = int(round(float(cpus))) # parse-n-round
        return available_processors



def get_available_processors_java_opts(procs):
    return "-XX:ActiveProcessorCount=%s" % procs


def get_gc_processors_java_opts(procs):
    return "-XX:ParallelGCThreads=%s" % procs


if __name__ == '__main__':
    procs = get_available_processors()
    args = get_java_args()
    args.append("-classpath")
    args.append(get_classpath())
    profile = get_profile()
    if profile:
        args.append(profile)
    yjp = get_yjp()
    if yjp:
        args.append(yjp)
    if procs:
        args.append(get_available_processors_java_opts(procs))
        args.append(get_gc_processors_java_opts(procs))
    args.append(MAIN_CLASS)
    os.chdir(get_base_dir())
    os.execv(JAVA_PATH, args)
