package ru.yandex.travel.hotels.extranet.cache

import org.springframework.boot.actuate.health.Health
import org.springframework.boot.actuate.health.HealthIndicator
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Component
import ru.yandex.travel.hotels.extranet.entities.HotelIdentifier
import ru.yandex.travel.yt_lucene_index.MapPersistentConfig
import javax.annotation.PostConstruct
import javax.annotation.PreDestroy
import ru.yandex.travel.hotels.administrator.export.proto.HotelConnection as HotelConnectionProto

@Component
@EnableConfigurationProperties(HotelConnectionServiceProperties::class)
class HotelConnectionService(
    params: HotelConnectionServiceProperties
) : HealthIndicator, IHotelConnectionService {
    private val mapPersistentConfig = if (params.enabled) {
        MapPersistentConfig(
            params.toYtLuceneIndexParams(),
            "HotelConnections",
            HotelConnectionProto::newBuilder,
            // external hotel id should be taken by default. The value is present for BNovo, it's the value that
            // hotel owner knows, hotelCode for Bnovo is something more specific for Yandex-Bnovo communication
            { HotelIdentifier(it.partnerId, it.externalHotelId.ifEmpty { it.hotelCode }) },
            HotelConnection::fromProto,
        )
    } else {
        null
    }

    override fun getByHotelIdKey(hotelIdentifier: HotelIdentifier, requestedEmail: String?): HotelConnection? =
        mapPersistentConfig?.getByKey(hotelIdentifier)

    @PostConstruct
    fun init() {
        mapPersistentConfig?.start()
    }

    @PreDestroy
    fun destroy() {
        mapPersistentConfig?.stop()
    }

    override fun health(): Health {
        return if (isReady()) {
            Health.up().build()
        } else {
            Health.down().build()
        }
    }

    private fun isReady(): Boolean {
        return mapPersistentConfig == null || mapPersistentConfig.isReady
    }
}
