package ru.yandex.travel.hotels.extranet.entities

import org.hibernate.annotations.CreationTimestamp
import org.hibernate.annotations.UpdateTimestamp
import ru.yandex.travel.hotels.extranet.validation.ValidAnnulationRules
import java.time.Instant
import javax.persistence.CascadeType
import javax.persistence.Entity
import javax.persistence.GeneratedValue
import javax.persistence.GenerationType
import javax.persistence.Id
import javax.persistence.ManyToOne
import javax.persistence.OneToMany
import javax.persistence.SequenceGenerator
import javax.persistence.Table
import javax.validation.Valid

@Entity
@Table(name = "annulation_policies")
class AnnulationPolicy private constructor(
    @ManyToOne
    val organization: Organization,

    var name: String,
) {
    constructor(organization: Organization, name: String, rules: List<AnnulationRule>) : this(organization, name) {
        rules.forEach { addRule(it) }
    }

    @ValidAnnulationRules
    @OneToMany(mappedBy = "policy", cascade = [CascadeType.ALL], orphanRemoval = true)
    @Valid
    val rules: MutableList<AnnulationRule> = mutableListOf()

    var deleted: Boolean = false

    @CreationTimestamp
    val createdAt: Instant = Instant.now()

    @UpdateTimestamp
    var updatedAt: Instant = Instant.now()

    @Id
    @GeneratedValue(
        strategy = GenerationType.SEQUENCE,
        generator = "annulation_policies_id_seq"
    )
    @SequenceGenerator(
        name = "annulation_policies_id_seq",
        sequenceName = "annulation_policies_id_seq",
        allocationSize = 1
    )
    val id: Long? = null

    fun addRule(rule: AnnulationRule) {
        require(rule.policy == null) { "Rule must be unbound" }
        rule.policy = this
        rules.add(rule)
    }

    fun describe(): String {
        return rules.sortedBy { it.comparator() }.reversed().joinToString(", затем ") { it.describe() }
    }
}
